/*
 This file is part of GNU Taler
 (C) 2024 GNUnet e.V.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import { AmountLike } from "./amounts.js";
import { canonicalJson } from "./helpers.js";
import {
  bufferFromAmount,
  bufferForUint64,
  buildSigPS,
  decodeCrock,
  EddsaPrivP,
  eddsaSign,
  EddsaSigP,
  hash,
  stringToBytes,
  timestampRoundedToBuffer,
} from "./taler-crypto.js";
import { TalerSignaturePurpose } from "./taler_signatures.js";
import { TalerProtocolTimestamp } from "./time.js";
import { AmlDecisionRequestWithoutSignature } from "./types-taler-exchange.js";

/**
 * Implementation of Taler protocol signatures.
 *
 * In this file, we have implementations of signatures that are not used in the wallet,
 * but in other places (tests, SPAs, ...).
 */

/**
 * Signature for the POST /aml/$OFFICER_PUB/decisions endpoint.
 */
export function signAmlDecision(
  priv: Uint8Array,
  decision: AmlDecisionRequestWithoutSignature,
): Uint8Array {
  const builder = buildSigPS(TalerSignaturePurpose.AML_DECISION);

  const flags: number = decision.keep_investigating ? 1 : 0;

  builder.put(timestampRoundedToBuffer(decision.decision_time));
  builder.put(
    timestampRoundedToBuffer(
      decision.attributes_expiration ?? TalerProtocolTimestamp.fromSeconds(0),
    ),
  );
  builder.put(decodeCrock(decision.h_payto));
  builder.put(hash(stringToBytes(decision.justification)));
  builder.put(hash(stringToBytes(canonicalJson(decision.properties) + "\0")));
  builder.put(hash(stringToBytes(canonicalJson(decision.new_rules) + "\0")));
  if (decision.new_measures != null) {
    builder.put(hash(stringToBytes(decision.new_measures)));
  } else {
    builder.put(new Uint8Array(64));
  }
  if (decision.attributes != null) {
    builder.put(hash(stringToBytes(canonicalJson(decision.attributes) + "\0")));
  } else {
    builder.put(new Uint8Array(64));
  }
  builder.put(bufferForUint64(flags));

  const sigBlob = builder.build();

  return eddsaSign(sigBlob, priv);
}

export function signAmlQuery(key: Uint8Array): EddsaSigP {
  const sigBlob = buildSigPS(TalerSignaturePurpose.AML_QUERY).build();

  return eddsaSign(sigBlob, key);
}

export function signWalletAccountSetup(
  key: EddsaPrivP,
  balance: AmountLike,
): EddsaSigP {
  const sigBlob = buildSigPS(TalerSignaturePurpose.WALLET_ACCOUNT_SETUP)
    .put(bufferFromAmount(balance))
    .build();

  return eddsaSign(sigBlob, key);
}

export function signKycAuth(key: EddsaPrivP): EddsaSigP {
  const sigBlob = buildSigPS(TalerSignaturePurpose.KYC_AUTH).build();

  return eddsaSign(sigBlob, key);
}
