/* readfile.c - Read the magic number, ID string, and specials in a gf
   or pk file.
   Copyright (C) 1997 Paul Vojta.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

/* Written by Paul Vojta.  */

#include <config.h>
#include <stdio.h>
#include <sys/stat.h>

#include "defs.h"
#include "modetype.h"

#define SPECIAL_MAX 1024	/* max length of a special that we can grok */

#define PK_MAGIC ((247 << 8) | 89)
#define GF_MAGIC ((247 << 8) | 131)

/* Data structure for storing strings from specials.  */
struct spcl
{
  /* Do we have it?  */
  boolean have;

  /* The string itself.  */
  char *str;

  /* How much space is allocated for it.  */
  unsigned int alloc_len;
};

/* Types for use in %S directives */
static const char spec_types[] = "jpgmbcfklo";

/* Names of those specials */
static const char *spec_names[] = {"jobname=", "pixels_per_inch=", "mag=",
				   "mode=", "fontfacebyte", "codingscheme=",
				   "fontid=", "blacker=", "fillin=",
				   "o_correction="};

#define SEQ_JOBNAME	0
#define SEQ_BDPI	1
#define SEQ_MAG		2
#define SEQ_MODE	3
#define SEQ_FONTFACEBYTE 4

#define NSPECIALS (sizeof spec_names / sizeof *spec_names)

/* The specials themselves.  */
struct spcl specials[NSPECIALS];

#ifndef atof
extern double atof();
#endif

#define one(fp) ((unsigned char) getc(fp))
static unsigned int two P_((FILE *fp));
static unsigned long four P_((FILE *fp));

/* Open the file and, if appropriate, read the magic number to determine the
   file format.  This routine assumes that file_status == FILE_NOWHERE.  */

void
open_file (relname, statp)
     char *relname;
     struct stat *statp;
{
  struct stat stat2;
  unsigned magic;

  mode_t st_mode = statp->st_mode;

  if (S_ISLNK(st_mode))
    if (stat(relname, &stat2) == 0)
      st_mode = stat2.st_mode;

  file_status = FILE_OPEN_FAILED;
  file_format = OTHER_FORMAT;

  if (!S_ISREG(st_mode))
    return;

  target_file = fopen(relname, "r");
  if (target_file == NULL)
    return;

  magic = one (target_file);
  magic = (magic << 8) | one (target_file);

  switch (magic)
    {
    case PK_MAGIC:
      file_format = PK_FORMAT;
      break;
    case GF_MAGIC:
      file_format = GF_FORMAT;
      break;
    }

  file_status = FILE_HAVE_FORMAT;
}

/* Read the initial identifying string of the file.  This routine assumes
   that file_status < FILE_HAVE_PRE.  */

void
read_file_pre (relname, statp)
     char *relname;
     struct stat *statp;
{
  unsigned len;
  char *p;

  if (file_status == FILE_NOWHERE)
    open_file (relname, statp);

  switch (file_format)
  {
  case OTHER_FORMAT:
    len = 0;
    break;
  case PK_FORMAT:
  case GF_FORMAT:
    len = one (target_file);
    break;
  }

  if (file_status != FILE_OPEN_FAILED)
    file_status = FILE_HAVE_PRE;

  if (file_pre_length <= len)
    {
      if (file_pre_length != 0)
	free (file_pre);
      do file_pre_length += 32;
      while (file_pre_length <= len);
      file_pre = xmalloc (file_pre_length);
    }

  if (len != 0)
    len = fread(file_pre, 1, len, target_file);
  file_pre[len] = '\0';
}

/* Routines for reading a given number of bytes from a file.  */

static unsigned int
two (fp)
     FILE *fp;
{
  unsigned int i;

  i = one(fp);
  return (i << 8) | one(fp);
}

static unsigned long
four (fp)
     FILE *fp;
{
  unsigned long i;

  i = two(fp);
  return (i << 16) | two(fp);
}

/* Read a special from the file and process it accordingly.  */

static void
read_a_special(len_len, yyy)
     unsigned int len_len;
     unsigned int yyy;
{
  unsigned long len;
  static char *special;
  static unsigned int special_alloc_len = 0;
  int i;

  len = 0;
  for (i = 0; i <= len_len; ++i)
    len = (len << 8) | one(target_file);

  /* Ignore specials that are too long.  */

  if (len > SPECIAL_MAX)
    {
      fseek(target_file, len, 1);
      return;
    }
  if (len == 0)
    return;

  if (len >= special_alloc_len)
    {
      if (special_alloc_len != 0)
	free (special);
      special_alloc_len = (len + 32) / 32 * 32;
      special = xmalloc (special_alloc_len);
    }

  len = fread(special, 1, len, target_file);
  special[len] = '\0';

  for (i = 0; i < NSPECIALS; ++i)
  {
    int len1 = strlen(spec_names[i]);

    if (len >= len1 && memcmp(special, spec_names[i], len1) == 0)
      {
	struct spcl *specp;

	if (i == SEQ_FONTFACEBYTE)
	  {
	    unsigned char c;
	    unsigned long l;

	    if (len != len1)
	      return;
	    c = one(target_file);
	    if (c != yyy)
	      {
		ungetc(c, target_file);
		return;
	      }
	    l = four(target_file);
	    if (l & 0xff00ffff)
	      {
		sprintf(special, "0x%08lx", l);
		len = 10;
	      }
	    else
	      {
		sprintf(special, "0x%02lx", l >> 16);
		len = 4;
	      }
	    len1 = 0;
	  }
	len -= len1 - 1;
	specp = specials + i;
	if (len > specp->alloc_len)
	  {
	    if (specp->alloc_len != 0)
	      free (specp->str);
	    specp->alloc_len = (len + 31) / 32 * 32;
	    specp->str = xmalloc (specp->alloc_len);
	  }
	memcpy(specp->str, special + len1, len);
	specp->have = true;
      }
  }
}

/* Read the specials from a PK file.  */

#define PK_ID      89
#define PK_CMD_START 240
#define PK_X1     240
#define PK_X2     241
#define PK_X3     242
#define PK_X4     243
#define PK_Y      244
#define PK_POST   245
#define PK_NOOP   246
#define PK_PRE    247

void
read_pk_specials P_((void))
{
  int PK_flag_byte;
  int flag_low_bits;
  long bytes_left;

  /* skip the initial four words */
  fseek (target_file, (long) 16, 1);

  for (;;)
    {
      do
	{
	  PK_flag_byte = one(target_file);
	  if (PK_flag_byte >= PK_CMD_START)
	    {
	      switch (PK_flag_byte)
	      {
	      case PK_X1:
	      case PK_X2:
	      case PK_X3:
	      case PK_X4:
		read_a_special (PK_flag_byte - PK_X1, PK_Y);
		break;
	      case PK_Y:
		(void) four(target_file);
		break;
	      case PK_POST :
	      case PK_NOOP :
		break;
	      default:
		return;
	      }
	    }
	}
      while (PK_flag_byte != PK_POST && PK_flag_byte >= PK_CMD_START);

      if (PK_flag_byte == PK_POST)
	break;
      flag_low_bits = PK_flag_byte & 0x7;
      if (flag_low_bits == 7)
	bytes_left = four(target_file) + 4;
      else if (flag_low_bits > 3)
	bytes_left = ((flag_low_bits - 4) << 16) + two(target_file) + 1;
      else
	bytes_left = (flag_low_bits << 8) + one(target_file) + 1;
      fseek (target_file, bytes_left, 1);
    }
}

/* Read the specials from a GF file.  */

#define	GF_ID_BYTE	131
#define	GF_XXX1		239
#define	GF_XXX2		240
#define	GF_XXX3		241
#define	GF_XXX4		242
#define	GF_YYY		243
#define	GF_POST		248
#define	GF_POST_POST	249
#define	GF_TRAILER	223		/* Trailing bytes at end of file */
#define	GF_TRAILER_4	((unsigned long) 223 << 24 | 223 << 16 | 223 << 8 | 223)

void
read_gf_specials P_((void))
{
  unsigned char ch;

  /* Find postamble.  */

  fseek(target_file, (long) -4, 2);
  while (four(target_file) != GF_TRAILER_4)
    fseek(target_file, (long) -5, 1);
  fseek(target_file, (long) -5, 1);
  for (;;)
    {
      ch = one(target_file);
      if (ch != GF_TRAILER) break;
      fseek(target_file, (long) -2, 1);
    }
  if (ch != GF_ID_BYTE)
    return;	/* something's wrong here.  */
  fseek(target_file, (long) -6, 1);
  if (one(target_file) != GF_POST_POST)
    return;	/* again, something's wrong.  */
  fseek(target_file, four(target_file), 0);	/* move to postamble */

  /* Read postamble.  */

  if (one(target_file) != GF_POST)
    return;
  fseek(target_file, four(target_file), 0);	/* move to last eoc + 1 */

  /* Now read the specials.  */

  for (;;)
    {
      ch = one(target_file);
      if (ch >= GF_XXX1 && ch <= GF_XXX4)
	read_a_special (ch - GF_XXX1, GF_YYY);
      else if (ch == GF_YYY)
	(void) four(target_file);
      else
	return;
    }
}

/* Read the identifying specials in the PK or GF file.  This routine assumes
   that file_status < FILE_HAVE_SPECIALS.  */

void
read_specials (relname, statp)
     char *relname;
     struct stat *statp;
{
  int i;

  if (file_status < FILE_HAVE_PRE)
    read_file_pre (relname, statp);

  if (file_status == FILE_OPEN_FAILED)
    return;

  for (i = 0; i < NSPECIALS; ++i)
    specials[i].have = false;
  file_status = FILE_HAVE_SPECIALS;

  switch (file_format)
  {
  case PK_FORMAT:
    read_pk_specials();
    break;
  case GF_FORMAT:
    read_gf_specials();
    break;
  }
}

/* Return true if we have specials for this file.  This routine assumes that
   file_format has been determined and is not OTHER_FORMAT.  */

boolean
have_specials (relname, statp)
     char *relname;
     struct stat *statp;
{
  if (file_status < FILE_HAVE_SPECIALS)
    read_specials (relname, statp);

  if (specials[SEQ_JOBNAME].have && specials[SEQ_BDPI].have
      && specials[SEQ_MAG].have && specials[SEQ_MODE].have)
    return true;

  return false;
}

/* Return true if the specials for the file agree with the file path.
   This routine assumes that file_format != OTHER_FORMAT.  */

boolean
check_specials (pathname, relname, statp)
     char *pathname;
     char *relname;
     struct stat *statp;
{
  char *p1;
  char *p2;
  char *p3;
  static char *ftype;
  int dpi;

  if (file_status < FILE_HAVE_SPECIALS)
    read_specials (relname, statp);

  /* Isolate the font name.  */
  p1 = strrchr (pathname, '/');
  if (p1 == NULL)
    p1 = pathname;
  else
    ++p1;
  p2 = strrchr (p1, '.');
  if (p2 == NULL)
    return false;

  /* Check the font name.  */
  if (specials[SEQ_JOBNAME].have)
    {
      int len = strlen(specials[SEQ_JOBNAME].str);

      if (p2 - p1 != len || memcmp (p1, specials[SEQ_JOBNAME].str, len) != 0)
	return false;
    }

  /* Determine the file format (according to its suffix).  */
  ++p2;
  p3 = p2;
  while (*p3 >= '0' && *p3 <= '9')
    ++p3;
  ftype = (file_format == GF_FORMAT ? "gf" : "pk");
  if (strcmp (p3, ftype) != 0)
    return false;

  /* If it's a dosnames file, isolate the dpi string.  */
  if (p2 == p3)	/* if no number between '.' and "pk" or "gf" */
    {
      if (p1 == pathname)
	{
	  static boolean warned_about_dpi = false;

	  if (!warned_about_dpi)
	    {
	      error (0, 0,
		   "warning - file path(s) too relative to get mode and dpi.");
	      warned_about_dpi = true;
	    }
	  return true;
	}
      p3 = p1 - 1;
      do
	--p1;
      while (p1 != pathname && p1[-1] != '/');
      if (memcmp (p1, "dpi", 3) != 0)
	return false;
      p2 = p1 + 3;
    }

  /* Compute the dpi.  */
  for (dpi = 0; p2 < p3; ++p2)
    {
      if (*p2 < '0' || *p2 > '9')
	return false;
      dpi = dpi * 10 + (*p2 - '0');
    }

  /* Check the dpi.  */
  if (specials[SEQ_BDPI].have)
    {
      float fdpi = atof(specials[SEQ_BDPI].str)
	         * (specials[SEQ_MAG].have ? atof(specials[SEQ_MAG].str) : 1.0);

      if (dpi < fdpi - .6 || dpi > fdpi + .6)
	return false;
    }

  if (specials[SEQ_MODE].have)
    {
      char *mode_value = specials[SEQ_MODE].str;
      int mode_len = strlen(mode_value);

      /* Find a directory named "pk" or "gf" (to find mode name).  */
      p2 = p3 = p1;
      for (;;)
	{
	  /* First, back up one directory */
	  if (p1 == pathname)
	    {
	      static boolean warned_about_mode = false;

	      if (!warned_about_mode)
		{
		  error(0, 0,
			"warning - file path(s) too relative to get mode.");
		  warned_about_mode = true;
		}
	      return true;
	    }
	  do
	    --p1;
	  while (p1 != pathname && p1[-1] != '/');

	  /* Is it "pk" or "gf"? */
	  if (p2 - p1 == 3
	      && (memcmp (p1, "pk", 2) == 0 || memcmp (p1, "gf", 2) == 0))
	    break;

	  p3 = p2;
	  p2 = p1;
	}

      if (memcmp(p1, ftype, 2) != 0)
	return false;	/* Sitting in wrong directory.  */
      if (p2 == p3)	/* if no room for a mode name */
	return false;
      --p3;

      if (strcmp(mode_value, "(ps2pk)Unknown mode") == 0
	  || strcmp(mode_value, "(gsftopk)Unknown") == 0)
	{
	  mode_value = "modeless";
	  mode_len = 8;
	}

      if (p3 - p2 != mode_len || memcmp (p2, mode_value, mode_len) != 0)
	return false;
    }

  return true;
}

/* Return a pointer to the character string for the appropriate special
   (or the preamble string).  */

char *
get_special (relname, statp, kind)
     char *relname;
     struct stat *statp;
     int kind;
{
  char *p;
  struct spcl *specp;

  if (kind == 'r') /* preamble comment */
    {
      if (file_status < FILE_HAVE_PRE)
	read_file_pre (relname, statp);
      return file_pre;
    }

  p = strchr(spec_types, kind);
  if (p != NULL)
    {
      if (file_status < FILE_HAVE_SPECIALS)
	read_specials (relname, statp);

      specp = &specials[p - spec_types];
      if (specp->have)
	return specp->str;
    }

  return "";
}
