/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

package org.apache.streampipes.processors.textmining.jvm.processor.sentencedetection;

import org.apache.streampipes.commons.exceptions.SpRuntimeException;
import org.apache.streampipes.extensions.api.pe.IStreamPipesDataProcessor;
import org.apache.streampipes.extensions.api.pe.config.IDataProcessorConfiguration;
import org.apache.streampipes.extensions.api.pe.context.EventProcessorRuntimeContext;
import org.apache.streampipes.extensions.api.pe.param.IDataProcessorParameters;
import org.apache.streampipes.extensions.api.pe.routing.SpOutputCollector;
import org.apache.streampipes.model.DataProcessorType;
import org.apache.streampipes.model.extensions.ExtensionAssetType;
import org.apache.streampipes.model.runtime.Event;
import org.apache.streampipes.model.schema.PropertyScope;
import org.apache.streampipes.sdk.builder.ProcessingElementBuilder;
import org.apache.streampipes.sdk.builder.StreamRequirementsBuilder;
import org.apache.streampipes.sdk.builder.processor.DataProcessorConfiguration;
import org.apache.streampipes.sdk.helpers.EpRequirements;
import org.apache.streampipes.sdk.helpers.Labels;
import org.apache.streampipes.sdk.helpers.Locales;
import org.apache.streampipes.sdk.helpers.OutputStrategies;

import opennlp.tools.sentdetect.SentenceDetectorME;
import opennlp.tools.sentdetect.SentenceModel;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

public class SentenceDetectionProcessor implements IStreamPipesDataProcessor {

  private static final String DETECTION_FIELD_KEY = "detectionField";
  private static final String BINARY_FILE_KEY = "binary-file";

  private String detection;
  private SentenceDetectorME sentenceDetector;

  @Override
  public IDataProcessorConfiguration declareConfig() {
    return DataProcessorConfiguration.create(
        SentenceDetectionProcessor::new,
        ProcessingElementBuilder
            .create("org.apache.streampipes.processors.textmining.jvm.sentencedetection", 0)
            .category(DataProcessorType.ENRICH_TEXT)
            .withAssets(ExtensionAssetType.DOCUMENTATION, ExtensionAssetType.ICON)
            .withLocales(Locales.EN)
            .requiredFile(Labels.withId(BINARY_FILE_KEY))
            .requiredStream(StreamRequirementsBuilder
                                .create()
                                .requiredPropertyWithUnaryMapping(
                                    EpRequirements.stringReq(),
                                    Labels.withId(DETECTION_FIELD_KEY),
                                    PropertyScope.NONE
                                )
                                .build())
            .outputStrategy(OutputStrategies.keep())
            .build()
    );
  }

  @Override
  public void onPipelineStarted(
      IDataProcessorParameters params,
      SpOutputCollector collector,
      EventProcessorRuntimeContext runtimeContext
  ) {
    String filename = params.extractor()
                            .selectedFilename(BINARY_FILE_KEY);
    byte[] fileContent = runtimeContext.getStreamPipesClient()
                                       .fileApi()
                                       .getFileContent(filename);
    this.detection = params.extractor()
                           .mappingPropertyValue(DETECTION_FIELD_KEY);

    InputStream modelIn = new ByteArrayInputStream(fileContent);
    SentenceModel model;
    try {
      model = new SentenceModel(modelIn);
    } catch (IOException e) {
      throw new SpRuntimeException("Error when loading the uploaded model.", e);
    }

    sentenceDetector = new SentenceDetectorME(model);
  }

  @Override
  public void onEvent(Event event, SpOutputCollector collector) throws SpRuntimeException {
    String text = event.getFieldBySelector(detection)
                       .getAsPrimitive()
                       .getAsString();

    String sentences[] = sentenceDetector.sentDetect(text);

    for (String sentence : sentences) {
      event.updateFieldBySelector(detection, sentence);
      collector.collect(event);
    }
  }

  @Override
  public void onPipelineStopped() {
  }
}
