/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * license agreements; and to You under the Apache License, version 2.0:
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * This file is part of the Apache Pekko project, which was derived from Akka.
 */

/*
 * Copyright (C) 2015-2022 Lightbend Inc. <https://www.lightbend.com>
 */

package jdocs.stream;

import java.util.Arrays;
import jdocs.AbstractJavaTest;
import org.apache.pekko.actor.ActorSystem;
import org.apache.pekko.stream.javadsl.Sink;
import org.apache.pekko.stream.javadsl.Source;
import org.apache.pekko.testkit.javadsl.TestKit;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

public class SubstreamDocTest extends AbstractJavaTest {

  static ActorSystem system;

  @BeforeClass
  public static void setup() {
    system = ActorSystem.create("FlowDocTest");
  }

  @AfterClass
  public static void tearDown() {
    TestKit.shutdownActorSystem(system);
    system = null;
  }

  @Test
  public void demonstrateGroupBy() throws Exception {
    // #groupBy1
    Source.from(Arrays.asList(1, 2, 3, 4, 5, 6, 7, 8, 9, 10)).groupBy(3, elem -> elem % 3);
    // #groupBy1

    // #groupBy2
    Source.from(Arrays.asList(1, 2, 3, 4, 5, 6, 7, 8, 9, 10))
        .groupBy(3, elem -> elem % 3)
        .to(Sink.ignore())
        .run(system);
    // #groupBy2

    // #groupBy3
    Source.from(Arrays.asList(1, 2, 3, 4, 5, 6, 7, 8, 9, 10))
        .groupBy(3, elem -> elem % 3)
        .mergeSubstreams()
        .runWith(Sink.ignore(), system);
    // #groupBy3

    // #groupBy4
    Source.from(Arrays.asList(1, 2, 3, 4, 5, 6, 7, 8, 9, 10))
        .groupBy(3, elem -> elem % 3)
        .mergeSubstreamsWithParallelism(2)
        .runWith(Sink.ignore(), system);
    // concatSubstreams is equivalent to mergeSubstreamsWithParallelism(1)
    Source.from(Arrays.asList(1, 2, 3, 4, 5, 6, 7, 8, 9, 10))
        .groupBy(3, elem -> elem % 3)
        .concatSubstreams()
        .runWith(Sink.ignore(), system);
    // #groupBy4
  }

  @Test
  public void demonstrateSplitWhenAfter() throws Exception {
    // #splitWhenAfter
    Source.from(Arrays.asList(1, 2, 3, 4, 5, 6, 7, 8, 9, 10)).splitWhen(elem -> elem == 3);

    Source.from(Arrays.asList(1, 2, 3, 4, 5, 6, 7, 8, 9, 10)).splitAfter(elem -> elem == 3);
    // #splitWhenAfter

    // #wordCount
    String text =
        "This is the first line.\n" + "The second line.\n" + "There is also the 3rd line\n";

    Source.from(Arrays.asList(text.split("")))
        .map(x -> x.charAt(0))
        .splitAfter(x -> x == '\n')
        .filter(x -> x != '\n')
        .map(x -> 1)
        .reduce((x, y) -> x + y)
        .to(Sink.foreach(x -> System.out.println(x)))
        .run(system);
    // #wordCount
    Thread.sleep(1000);
  }

  @Test
  public void demonstrateflatMapConcatMerge() throws Exception {
    // #flatMapConcat
    Source.from(Arrays.asList(1, 2))
        .flatMapConcat(i -> Source.from(Arrays.asList(i, i, i)))
        .runWith(Sink.ignore(), system);
    // #flatMapConcat

    // #flatMapMerge
    Source.from(Arrays.asList(1, 2))
        .flatMapMerge(2, i -> Source.from(Arrays.asList(i, i, i)))
        .runWith(Sink.ignore(), system);
    // #flatMapMerge
  }
}
