/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.replication;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.time.Clock;
import java.time.Instant;
import java.time.ZoneId;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.OptionalLong;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.IntConsumer;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.metrics2.lib.MetricsRegistry;
import org.apache.hadoop.metrics2.lib.MutableRate;
import org.apache.hadoop.ozone.container.common.statemachine.DatanodeConfiguration;
import org.apache.hadoop.ozone.container.common.statemachine.StateContext;
import org.apache.hadoop.ozone.container.replication.AbstractReplicationTask;
import org.apache.hadoop.ozone.container.replication.ReplicationServer;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ReplicationSupervisor {
    private static final Logger LOG = LoggerFactory.getLogger(ReplicationSupervisor.class);
    private static final Comparator<TaskRunner> TASK_RUNNER_COMPARATOR = Comparator.comparing(TaskRunner::getTaskPriority).thenComparing(TaskRunner::getTaskQueueTime);
    private final ExecutorService executor;
    private final StateContext context;
    private final Clock clock;
    private final Map<String, AtomicLong> requestCounter = new ConcurrentHashMap<String, AtomicLong>();
    private final Map<String, AtomicLong> successCounter = new ConcurrentHashMap<String, AtomicLong>();
    private final Map<String, AtomicLong> failureCounter = new ConcurrentHashMap<String, AtomicLong>();
    private final Map<String, AtomicLong> timeoutCounter = new ConcurrentHashMap<String, AtomicLong>();
    private final Map<String, AtomicLong> skippedCounter = new ConcurrentHashMap<String, AtomicLong>();
    private final Map<String, AtomicLong> queuedCounter = new ConcurrentHashMap<String, AtomicLong>();
    private final MetricsRegistry registry;
    private final Map<String, MutableRate> opsLatencyMs = new ConcurrentHashMap<String, MutableRate>();
    private static final Map<String, String> METRICS_MAP = new HashMap<String, String>();
    private final Set<AbstractReplicationTask> inFlight;
    private final Map<Class<?>, AtomicInteger> taskCounter = new ConcurrentHashMap();
    private int maxQueueSize;
    private final AtomicReference<HddsProtos.NodeOperationalState> state = new AtomicReference();
    private final IntConsumer executorThreadUpdater;
    private final ReplicationServer.ReplicationConfig replicationConfig;
    private final DatanodeConfiguration datanodeConfig;

    public static Builder newBuilder() {
        return new Builder();
    }

    public static Map<String, String> getMetricsMap() {
        return Collections.unmodifiableMap(METRICS_MAP);
    }

    private ReplicationSupervisor(StateContext context, ExecutorService executor, ReplicationServer.ReplicationConfig replicationConfig, DatanodeConfiguration datanodeConfig, Clock clock, IntConsumer executorThreadUpdater) {
        DatanodeDetails dn;
        this.inFlight = ConcurrentHashMap.newKeySet();
        this.context = context;
        this.executor = executor;
        this.replicationConfig = replicationConfig;
        this.datanodeConfig = datanodeConfig;
        this.maxQueueSize = datanodeConfig.getCommandQueueLimit();
        this.clock = clock;
        this.executorThreadUpdater = executorThreadUpdater;
        if (context != null && (dn = context.getParent().getDatanodeDetails()) != null) {
            this.nodeStateUpdated(dn.getPersistedOpState());
        }
        this.registry = new MetricsRegistry(ReplicationSupervisor.class.getSimpleName());
    }

    public void addTask(AbstractReplicationTask task) {
        if (this.queueHasRoomFor(task)) {
            this.initCounters(task);
            this.addToQueue(task);
        }
    }

    private boolean queueHasRoomFor(AbstractReplicationTask task) {
        int max = this.maxQueueSize;
        if (this.getTotalInFlightReplications() >= max) {
            LOG.warn("Ignored {} command for container {} in Replication Supervisoras queue reached max size of {}.", new Object[]{task.getClass(), task.getContainerId(), max});
            return false;
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void initCounters(AbstractReplicationTask task) {
        if (this.requestCounter.get(task.getMetricName()) == null) {
            ReplicationSupervisor replicationSupervisor = this;
            synchronized (replicationSupervisor) {
                if (this.requestCounter.get(task.getMetricName()) == null) {
                    this.requestCounter.put(task.getMetricName(), new AtomicLong(0L));
                    this.successCounter.put(task.getMetricName(), new AtomicLong(0L));
                    this.failureCounter.put(task.getMetricName(), new AtomicLong(0L));
                    this.timeoutCounter.put(task.getMetricName(), new AtomicLong(0L));
                    this.skippedCounter.put(task.getMetricName(), new AtomicLong(0L));
                    this.queuedCounter.put(task.getMetricName(), new AtomicLong(0L));
                    this.opsLatencyMs.put(task.getMetricName(), this.registry.newRate(task.getClass().getSimpleName() + "Ms"));
                    METRICS_MAP.put(task.getMetricName(), task.getMetricDescriptionSegment());
                }
            }
        }
    }

    private void addToQueue(AbstractReplicationTask task) {
        if (this.inFlight.add(task)) {
            if (task.getPriority() != StorageContainerDatanodeProtocolProtos.ReplicationCommandPriority.LOW) {
                this.taskCounter.computeIfAbsent(task.getClass(), k -> new AtomicInteger()).incrementAndGet();
            }
            this.queuedCounter.get(task.getMetricName()).incrementAndGet();
            this.executor.execute(new TaskRunner(task));
        }
    }

    private void decrementTaskCounter(AbstractReplicationTask task) {
        if (task.getPriority() == StorageContainerDatanodeProtocolProtos.ReplicationCommandPriority.LOW) {
            return;
        }
        AtomicInteger counter = this.taskCounter.get(task.getClass());
        if (counter != null) {
            counter.decrementAndGet();
        }
    }

    @VisibleForTesting
    public void shutdownAfterFinish() throws InterruptedException {
        this.executor.shutdown();
        this.executor.awaitTermination(1L, TimeUnit.DAYS);
    }

    public void stop() {
        try {
            this.executor.shutdown();
            if (!this.executor.awaitTermination(3L, TimeUnit.SECONDS)) {
                this.executor.shutdownNow();
            }
        }
        catch (InterruptedException ie) {
            Thread.currentThread().interrupt();
        }
    }

    public int getInFlightReplications(Class<? extends AbstractReplicationTask> taskClass) {
        AtomicInteger counter = this.taskCounter.get(taskClass);
        return counter == null ? 0 : counter.get();
    }

    public Map<String, Integer> getInFlightReplicationSummary() {
        HashMap<String, Integer> result = new HashMap<String, Integer>();
        for (Map.Entry<Class<?>, AtomicInteger> entry : this.taskCounter.entrySet()) {
            result.put(entry.getKey().getSimpleName(), entry.getValue().get());
        }
        return result;
    }

    public int getTotalInFlightReplications() {
        return this.inFlight.size();
    }

    public int getMaxQueueSize() {
        return this.maxQueueSize;
    }

    public void nodeStateUpdated(HddsProtos.NodeOperationalState newState) {
        if (this.state.getAndSet(newState) != newState) {
            int threadCount = this.replicationConfig.getReplicationMaxStreams();
            int newMaxQueueSize = this.datanodeConfig.getCommandQueueLimit();
            if (DatanodeDetails.isMaintenance((HddsProtos.NodeOperationalState)newState) || DatanodeDetails.isDecommission((HddsProtos.NodeOperationalState)newState)) {
                threadCount = this.replicationConfig.scaleOutOfServiceLimit(threadCount);
                newMaxQueueSize = this.replicationConfig.scaleOutOfServiceLimit(newMaxQueueSize);
            }
            LOG.info("Node state updated to {}, scaling executor pool size to {}", (Object)newState, (Object)threadCount);
            this.maxQueueSize = newMaxQueueSize;
            this.executorThreadUpdater.accept(threadCount);
        }
    }

    public long getReplicationRequestCount() {
        return this.getCount(this.requestCounter);
    }

    public long getReplicationRequestCount(String metricsName) {
        AtomicLong counter = this.requestCounter.get(metricsName);
        return counter != null ? counter.get() : 0L;
    }

    public long getQueueSize() {
        if (this.executor instanceof ThreadPoolExecutor) {
            return ((ThreadPoolExecutor)this.executor).getQueue().size();
        }
        return 0L;
    }

    public long getMaxReplicationStreams() {
        if (this.executor instanceof ThreadPoolExecutor) {
            return ((ThreadPoolExecutor)this.executor).getMaximumPoolSize();
        }
        return 1L;
    }

    private long getCount(Map<String, AtomicLong> counter) {
        long total = 0L;
        for (Map.Entry<String, AtomicLong> entry : counter.entrySet()) {
            total += entry.getValue().get();
        }
        return total;
    }

    public long getReplicationSuccessCount() {
        return this.getCount(this.successCounter);
    }

    public long getReplicationSuccessCount(String metricsName) {
        AtomicLong counter = this.successCounter.get(metricsName);
        return counter != null ? counter.get() : 0L;
    }

    public long getReplicationFailureCount() {
        return this.getCount(this.failureCounter);
    }

    public long getReplicationFailureCount(String metricsName) {
        AtomicLong counter = this.failureCounter.get(metricsName);
        return counter != null ? counter.get() : 0L;
    }

    public long getReplicationTimeoutCount() {
        return this.getCount(this.timeoutCounter);
    }

    public long getReplicationTimeoutCount(String metricsName) {
        AtomicLong counter = this.timeoutCounter.get(metricsName);
        return counter != null ? counter.get() : 0L;
    }

    public long getReplicationSkippedCount() {
        return this.getCount(this.skippedCounter);
    }

    public long getReplicationSkippedCount(String metricsName) {
        AtomicLong counter = this.skippedCounter.get(metricsName);
        return counter != null ? counter.get() : 0L;
    }

    public long getReplicationQueuedCount() {
        return this.getCount(this.queuedCounter);
    }

    public long getReplicationQueuedCount(String metricsName) {
        AtomicLong counter = this.queuedCounter.get(metricsName);
        return counter != null ? counter.get() : 0L;
    }

    public long getReplicationRequestAvgTime(String metricsName) {
        MutableRate rate = this.opsLatencyMs.get(metricsName);
        return rate != null ? (long)Math.ceil(rate.lastStat().mean()) : 0L;
    }

    public long getReplicationRequestTotalTime(String metricsName) {
        MutableRate rate = this.opsLatencyMs.get(metricsName);
        return rate != null ? (long)Math.ceil(rate.lastStat().total()) : 0L;
    }

    public static class Builder {
        private StateContext context;
        private ReplicationServer.ReplicationConfig replicationConfig;
        private DatanodeConfiguration datanodeConfig;
        private ExecutorService executor;
        private Clock clock;
        private IntConsumer executorThreadUpdater = threadCount -> {};

        public Builder clock(Clock newClock) {
            this.clock = newClock;
            return this;
        }

        public Builder executor(ExecutorService newExecutor) {
            this.executor = newExecutor;
            return this;
        }

        public Builder replicationConfig(ReplicationServer.ReplicationConfig newReplicationConfig) {
            this.replicationConfig = newReplicationConfig;
            return this;
        }

        public Builder datanodeConfig(DatanodeConfiguration newDatanodeConfig) {
            this.datanodeConfig = newDatanodeConfig;
            return this;
        }

        public Builder stateContext(StateContext newContext) {
            this.context = newContext;
            return this;
        }

        public Builder executorThreadUpdater(IntConsumer newUpdater) {
            this.executorThreadUpdater = newUpdater;
            return this;
        }

        public ReplicationSupervisor build() {
            if (this.replicationConfig == null || this.datanodeConfig == null) {
                OzoneConfiguration conf = new OzoneConfiguration();
                if (this.replicationConfig == null) {
                    this.replicationConfig = (ReplicationServer.ReplicationConfig)conf.getObject(ReplicationServer.ReplicationConfig.class);
                }
                if (this.datanodeConfig == null) {
                    this.datanodeConfig = (DatanodeConfiguration)((Object)conf.getObject(DatanodeConfiguration.class));
                }
            }
            if (this.clock == null) {
                this.clock = Clock.system(ZoneId.systemDefault());
            }
            if (this.executor == null) {
                LOG.info("Initializing replication supervisor with thread count = {}", (Object)this.replicationConfig.getReplicationMaxStreams());
                String threadNamePrefix = this.context != null ? this.context.getThreadNamePrefix() : "";
                ThreadFactory threadFactory = new ThreadFactoryBuilder().setDaemon(true).setNameFormat(threadNamePrefix + "ContainerReplicationThread-%d").build();
                ThreadPoolExecutor tpe = new ThreadPoolExecutor(this.replicationConfig.getReplicationMaxStreams(), this.replicationConfig.getReplicationMaxStreams(), 60L, TimeUnit.SECONDS, new PriorityBlockingQueue<Runnable>(), threadFactory);
                this.executor = tpe;
                this.executorThreadUpdater = threadCount -> {
                    if (threadCount < tpe.getCorePoolSize()) {
                        tpe.setCorePoolSize(threadCount);
                        tpe.setMaximumPoolSize(threadCount);
                    } else {
                        tpe.setMaximumPoolSize(threadCount);
                        tpe.setCorePoolSize(threadCount);
                    }
                };
            }
            return new ReplicationSupervisor(this.context, this.executor, this.replicationConfig, this.datanodeConfig, this.clock, this.executorThreadUpdater);
        }
    }

    public final class TaskRunner
    implements Comparable<TaskRunner>,
    Runnable {
        private final AbstractReplicationTask task;

        public TaskRunner(AbstractReplicationTask task) {
            this.task = task;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            long startTime = Time.monotonicNow();
            try {
                ((AtomicLong)ReplicationSupervisor.this.requestCounter.get(this.task.getMetricName())).incrementAndGet();
                long now = ReplicationSupervisor.this.clock.millis();
                long deadline = this.task.getDeadline();
                if (deadline > 0L && now > deadline) {
                    LOG.info("Ignoring {} since the deadline has passed ({} < {})", new Object[]{this, Instant.ofEpochMilli(deadline), Instant.ofEpochMilli(now)});
                    ((AtomicLong)ReplicationSupervisor.this.timeoutCounter.get(this.task.getMetricName())).incrementAndGet();
                    return;
                }
                if (ReplicationSupervisor.this.context != null) {
                    DatanodeDetails dn = ReplicationSupervisor.this.context.getParent().getDatanodeDetails();
                    if (dn != null && dn.getPersistedOpState() != HddsProtos.NodeOperationalState.IN_SERVICE && this.task.shouldOnlyRunOnInServiceDatanodes()) {
                        LOG.info("Ignoring {} since datanode is not in service ({})", (Object)this, (Object)dn.getPersistedOpState());
                        return;
                    }
                    OptionalLong currentTerm = ReplicationSupervisor.this.context.getTermOfLeaderSCM();
                    long taskTerm = this.task.getTerm();
                    if (currentTerm.isPresent() && taskTerm < currentTerm.getAsLong()) {
                        LOG.info("Ignoring {} since SCM leader has new term ({} < {})", new Object[]{this, taskTerm, currentTerm.getAsLong()});
                        return;
                    }
                }
                this.task.setStatus(AbstractReplicationTask.Status.IN_PROGRESS);
                this.task.runTask();
                if (this.task.getStatus() == AbstractReplicationTask.Status.FAILED) {
                    LOG.warn("Failed {}", (Object)this);
                    ((AtomicLong)ReplicationSupervisor.this.failureCounter.get(this.task.getMetricName())).incrementAndGet();
                } else if (this.task.getStatus() == AbstractReplicationTask.Status.DONE) {
                    LOG.info("Successful {}", (Object)this);
                    ((AtomicLong)ReplicationSupervisor.this.successCounter.get(this.task.getMetricName())).incrementAndGet();
                } else if (this.task.getStatus() == AbstractReplicationTask.Status.SKIPPED) {
                    LOG.info("Skipped {}", (Object)this);
                    ((AtomicLong)ReplicationSupervisor.this.skippedCounter.get(this.task.getMetricName())).incrementAndGet();
                }
            }
            catch (Exception e) {
                this.task.setStatus(AbstractReplicationTask.Status.FAILED);
                LOG.warn("Failed {}", (Object)this, (Object)e);
                ((AtomicLong)ReplicationSupervisor.this.failureCounter.get(this.task.getMetricName())).incrementAndGet();
            }
            finally {
                ((AtomicLong)ReplicationSupervisor.this.queuedCounter.get(this.task.getMetricName())).decrementAndGet();
                ((MutableRate)ReplicationSupervisor.this.opsLatencyMs.get(this.task.getMetricName())).add(Time.monotonicNow() - startTime);
                ReplicationSupervisor.this.inFlight.remove(this.task);
                ReplicationSupervisor.this.decrementTaskCounter(this.task);
            }
        }

        public String toString() {
            return this.task.toString();
        }

        public StorageContainerDatanodeProtocolProtos.ReplicationCommandPriority getTaskPriority() {
            return this.task.getPriority();
        }

        public long getTaskQueueTime() {
            return this.task.getQueued().toEpochMilli();
        }

        @Override
        public int compareTo(TaskRunner o) {
            return TASK_RUNNER_COMPARATOR.compare(this, o);
        }

        public int hashCode() {
            return Objects.hash(this.task);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            TaskRunner that = (TaskRunner)o;
            return this.task.equals(that.task);
        }
    }
}

