/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.checksum;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.Striped;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.StandardCopyOption;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.BiConsumer;
import java.util.function.Function;
import org.apache.hadoop.hdds.HddsUtils;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.hdds.utils.SimpleStriped;
import org.apache.hadoop.metrics2.lib.MutableRate;
import org.apache.hadoop.ozone.container.checksum.ContainerDiffReport;
import org.apache.hadoop.ozone.container.checksum.ContainerMerkleTreeMetrics;
import org.apache.hadoop.ozone.container.checksum.ContainerMerkleTreeWriter;
import org.apache.hadoop.ozone.container.common.helpers.BlockData;
import org.apache.hadoop.ozone.container.common.impl.ContainerData;
import org.apache.hadoop.ozone.container.common.statemachine.DatanodeConfiguration;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainerData;
import org.apache.hadoop.ozone.util.MetricUtil;
import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import org.apache.ratis.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContainerChecksumTreeManager {
    private static final Logger LOG = LoggerFactory.getLogger(ContainerChecksumTreeManager.class);
    private final Striped<Lock> fileLocks;
    private final ContainerMerkleTreeMetrics metrics;

    public ContainerChecksumTreeManager(ConfigurationSource conf) {
        this.fileLocks = SimpleStriped.custom((int)((DatanodeConfiguration)((Object)conf.getObject(DatanodeConfiguration.class))).getContainerChecksumLockStripes(), () -> new ReentrantLock(true));
        this.metrics = ContainerMerkleTreeMetrics.create();
    }

    public void stop() {
        ContainerMerkleTreeMetrics.unregister();
    }

    public ContainerDiffReport diff(ContainerProtos.ContainerChecksumInfo thisChecksumInfo, ContainerProtos.ContainerChecksumInfo peerChecksumInfo) throws StorageContainerException {
        ContainerDiffReport report = new ContainerDiffReport(thisChecksumInfo.getContainerID());
        try {
            Preconditions.assertNotNull((Object)thisChecksumInfo, (String)"Datanode's checksum info is null.");
            Preconditions.assertNotNull((Object)peerChecksumInfo, (String)"Peer checksum info is null.");
            if (thisChecksumInfo.getContainerID() != peerChecksumInfo.getContainerID()) {
                throw new StorageContainerException("Container ID does not match. Local container ID " + thisChecksumInfo.getContainerID() + " , Peer container ID " + peerChecksumInfo.getContainerID(), ContainerProtos.Result.CONTAINER_ID_MISMATCH);
            }
            MetricUtil.captureLatencyNs((MutableRate)this.metrics.getMerkleTreeDiffLatencyNS(), () -> this.compareContainerMerkleTree(thisChecksumInfo, peerChecksumInfo, report));
        }
        catch (IOException ex) {
            this.metrics.incrementMerkleTreeDiffFailures();
            throw new StorageContainerException("Container Diff failed for container #" + thisChecksumInfo.getContainerID(), (Throwable)ex, ContainerProtos.Result.IO_EXCEPTION);
        }
        if (report.needsRepair()) {
            this.metrics.incrementRepairContainerDiffs();
            this.metrics.incrementCorruptChunksIdentified(report.getNumCorruptChunks());
            this.metrics.incrementMissingBlocksIdentified(report.getNumMissingBlocks());
            this.metrics.incrementMissingChunksIdentified(report.getNumMissingChunks());
            this.metrics.incrementDivergedDeletedBlocksIdentified(report.getNumdivergedDeletedBlocks());
        } else {
            this.metrics.incrementNoRepairContainerDiffs();
        }
        return report;
    }

    private void compareContainerMerkleTree(ContainerProtos.ContainerChecksumInfo thisChecksumInfo, ContainerProtos.ContainerChecksumInfo peerChecksumInfo, ContainerDiffReport report) {
        ContainerProtos.ContainerMerkleTree thisMerkleTree = thisChecksumInfo.getContainerMerkleTree();
        ContainerProtos.ContainerMerkleTree peerMerkleTree = peerChecksumInfo.getContainerMerkleTree();
        if (thisMerkleTree.getDataChecksum() == peerMerkleTree.getDataChecksum()) {
            return;
        }
        List thisBlockMerkleTreeList = thisMerkleTree.getBlockMerkleTreeList();
        List peerBlockMerkleTreeList = peerMerkleTree.getBlockMerkleTreeList();
        int thisIdx = 0;
        int peerIdx = 0;
        while (thisIdx < thisBlockMerkleTreeList.size() && peerIdx < peerBlockMerkleTreeList.size()) {
            ContainerProtos.BlockMerkleTree thisBlockMerkleTree = (ContainerProtos.BlockMerkleTree)thisBlockMerkleTreeList.get(thisIdx);
            ContainerProtos.BlockMerkleTree peerBlockMerkleTree = (ContainerProtos.BlockMerkleTree)peerBlockMerkleTreeList.get(peerIdx);
            if (thisBlockMerkleTree.getBlockID() == peerBlockMerkleTree.getBlockID()) {
                if (thisBlockMerkleTree.getDataChecksum() != peerBlockMerkleTree.getDataChecksum()) {
                    this.compareBlockMerkleTree(thisBlockMerkleTree, peerBlockMerkleTree, report);
                }
                ++thisIdx;
                ++peerIdx;
                continue;
            }
            if (thisBlockMerkleTree.getBlockID() < peerBlockMerkleTree.getBlockID()) {
                ++thisIdx;
                continue;
            }
            if (peerBlockMerkleTree.getDeleted()) {
                report.addDivergedDeletedBlock(peerBlockMerkleTree);
            } else {
                report.addMissingBlock(peerBlockMerkleTree);
            }
            ++peerIdx;
        }
        while (peerIdx < peerBlockMerkleTreeList.size()) {
            ContainerProtos.BlockMerkleTree peerBlockMerkleTree = (ContainerProtos.BlockMerkleTree)peerBlockMerkleTreeList.get(peerIdx);
            if (peerBlockMerkleTree.getDeleted()) {
                report.addDivergedDeletedBlock(peerBlockMerkleTree);
            } else {
                report.addMissingBlock(peerBlockMerkleTree);
            }
            ++peerIdx;
        }
    }

    private void compareBlockMerkleTree(ContainerProtos.BlockMerkleTree thisBlockMerkleTree, ContainerProtos.BlockMerkleTree peerBlockMerkleTree, ContainerDiffReport report) {
        boolean thisBlockDeleted = thisBlockMerkleTree.getDeleted();
        boolean peerBlockDeleted = peerBlockMerkleTree.getDeleted();
        if (thisBlockDeleted) {
            if (peerBlockDeleted && thisBlockMerkleTree.getDataChecksum() < peerBlockMerkleTree.getDataChecksum()) {
                report.addDivergedDeletedBlock(peerBlockMerkleTree);
            }
        } else if (peerBlockDeleted) {
            report.addDivergedDeletedBlock(peerBlockMerkleTree);
        } else {
            this.compareChunkMerkleTrees(thisBlockMerkleTree, peerBlockMerkleTree, report);
        }
    }

    private void compareChunkMerkleTrees(ContainerProtos.BlockMerkleTree thisBlockMerkleTree, ContainerProtos.BlockMerkleTree peerBlockMerkleTree, ContainerDiffReport report) {
        List thisChunkMerkleTreeList = thisBlockMerkleTree.getChunkMerkleTreeList();
        List peerChunkMerkleTreeList = peerBlockMerkleTree.getChunkMerkleTreeList();
        int thisIdx = 0;
        int peerIdx = 0;
        long containerID = report.getContainerID();
        long blockID = thisBlockMerkleTree.getBlockID();
        while (thisIdx < thisChunkMerkleTreeList.size() && peerIdx < peerChunkMerkleTreeList.size()) {
            ContainerProtos.ChunkMerkleTree thisChunkMerkleTree = (ContainerProtos.ChunkMerkleTree)thisChunkMerkleTreeList.get(thisIdx);
            ContainerProtos.ChunkMerkleTree peerChunkMerkleTree = (ContainerProtos.ChunkMerkleTree)peerChunkMerkleTreeList.get(peerIdx);
            if (thisChunkMerkleTree.getOffset() == peerChunkMerkleTree.getOffset()) {
                if (thisChunkMerkleTree.getDataChecksum() != peerChunkMerkleTree.getDataChecksum() && !thisChunkMerkleTree.getChecksumMatches()) {
                    this.reportChunkIfHealthy(containerID, blockID, peerChunkMerkleTree, report::addCorruptChunk);
                }
                ++thisIdx;
                ++peerIdx;
                continue;
            }
            if (thisChunkMerkleTree.getOffset() < peerChunkMerkleTree.getOffset()) {
                ++thisIdx;
                continue;
            }
            this.reportChunkIfHealthy(containerID, blockID, peerChunkMerkleTree, report::addMissingChunk);
            ++peerIdx;
        }
        while (peerIdx < peerChunkMerkleTreeList.size()) {
            this.reportChunkIfHealthy(containerID, blockID, (ContainerProtos.ChunkMerkleTree)peerChunkMerkleTreeList.get(peerIdx), report::addMissingChunk);
            ++peerIdx;
        }
    }

    private void reportChunkIfHealthy(long containerID, long blockID, ContainerProtos.ChunkMerkleTree peerTree, BiConsumer<Long, ContainerProtos.ChunkMerkleTree> addToReport) {
        if (peerTree.getChecksumMatches()) {
            addToReport.accept(blockID, peerTree);
        } else {
            LOG.warn("Skipping chunk at offset {} in block {} of container {} since peer reported it as unhealthy.", new Object[]{peerTree.getOffset(), blockID, containerID});
        }
    }

    public static long getDataChecksum(ContainerProtos.ContainerChecksumInfo checksumInfo) {
        return checksumInfo.getContainerMerkleTree().getDataChecksum();
    }

    public static File getContainerChecksumFile(ContainerData data) {
        return new File(data.getMetadataPath(), data.getContainerID() + ".tree");
    }

    public static boolean hasDataChecksum(ContainerProtos.ContainerChecksumInfo checksumInfo) {
        return checksumInfo != null && checksumInfo.hasContainerMerkleTree() && checksumInfo.getContainerMerkleTree().hasDataChecksum();
    }

    @VisibleForTesting
    public static File getTmpContainerChecksumFile(ContainerData data) {
        return new File(data.getMetadataPath(), data.getContainerID() + ".tree" + ".tmp");
    }

    private Lock getLock(long containerID) {
        return (Lock)this.fileLocks.get((Object)containerID);
    }

    public ContainerProtos.ContainerChecksumInfo read(ContainerData data) throws IOException {
        try {
            return (ContainerProtos.ContainerChecksumInfo)MetricUtil.captureLatencyNs((MutableRate)this.metrics.getReadContainerMerkleTreeLatencyNS(), () -> ContainerChecksumTreeManager.readChecksumInfo(data));
        }
        catch (IOException ex) {
            this.metrics.incrementMerkleTreeReadFailures();
            throw ex;
        }
    }

    public ContainerProtos.ContainerChecksumInfo updateTree(ContainerData data, ContainerMerkleTreeWriter treeWriter) throws IOException {
        return this.write(data, treeWriter::update);
    }

    public void addDeletedBlocks(ContainerData data, Collection<BlockData> blocks) throws IOException {
        this.write(data, existingTree -> {
            ContainerMerkleTreeWriter treeWriter = new ContainerMerkleTreeWriter((ContainerProtos.ContainerMerkleTree)existingTree);
            return treeWriter.addDeletedBlocks(blocks, existingTree.hasDataChecksum());
        });
    }

    private ContainerProtos.ContainerChecksumInfo readOrCreate(ContainerData data) {
        try {
            return this.read(data);
        }
        catch (IOException ex) {
            LOG.error("Failed to read container checksum tree file for container {}. Overwriting it with a new instance.", (Object)data.getContainerID(), (Object)ex);
            return ContainerProtos.ContainerChecksumInfo.newBuilder().build();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ContainerProtos.ContainerChecksumInfo write(ContainerData data, Function<ContainerProtos.ContainerMerkleTree, ContainerProtos.ContainerMerkleTree> mergeFunction) throws IOException {
        long containerID = data.getContainerID();
        Lock fileLock = this.getLock(containerID);
        fileLock.lock();
        try {
            ContainerProtos.ContainerChecksumInfo currentChecksumInfo = this.readOrCreate(data);
            ContainerProtos.ContainerChecksumInfo.Builder newChecksumInfoBuilder = currentChecksumInfo.toBuilder();
            ContainerProtos.ContainerMerkleTree treeProto = (ContainerProtos.ContainerMerkleTree)MetricUtil.captureLatencyNs((MutableRate)this.metrics.getCreateMerkleTreeLatencyNS(), () -> (ContainerProtos.ContainerMerkleTree)mergeFunction.apply(currentChecksumInfo.getContainerMerkleTree()));
            ContainerProtos.ContainerChecksumInfo newChecksumInfo = newChecksumInfoBuilder.setContainerMerkleTree(treeProto).setContainerID(containerID).build();
            File checksumFile = ContainerChecksumTreeManager.getContainerChecksumFile(data);
            File tmpChecksumFile = ContainerChecksumTreeManager.getTmpContainerChecksumFile(data);
            try (OutputStream tmpOutputStream = Files.newOutputStream(tmpChecksumFile.toPath(), new OpenOption[0]);){
                MetricUtil.captureLatencyNs((MutableRate)this.metrics.getWriteContainerMerkleTreeLatencyNS(), () -> {
                    newChecksumInfo.writeTo(tmpOutputStream);
                    Files.move(tmpChecksumFile.toPath(), checksumFile.toPath(), StandardCopyOption.ATOMIC_MOVE);
                });
                LOG.debug("Merkle tree for container {} updated with container data checksum {}", (Object)containerID, (Object)HddsUtils.checksumToString((long)treeProto.getDataChecksum()));
            }
            catch (IOException ex) {
                this.metrics.incrementMerkleTreeWriteFailures();
                throw new IOException("Error occurred when writing container merkle tree for containerID " + data.getContainerID(), ex);
            }
            ContainerProtos.ContainerChecksumInfo containerChecksumInfo = newChecksumInfo;
            return containerChecksumInfo;
        }
        finally {
            fileLock.unlock();
        }
    }

    public ByteString getContainerChecksumInfo(KeyValueContainerData data) throws IOException {
        File checksumFile = ContainerChecksumTreeManager.getContainerChecksumFile(data);
        if (!checksumFile.exists()) {
            throw new FileNotFoundException("Checksum file does not exist for container #" + data.getContainerID());
        }
        try (InputStream inStream = Files.newInputStream(checksumFile.toPath(), new OpenOption[0]);){
            ByteString byteString = ByteString.readFrom((InputStream)inStream);
            return byteString;
        }
    }

    public static ContainerProtos.ContainerChecksumInfo readChecksumInfo(ContainerData data) throws IOException {
        ContainerProtos.ContainerChecksumInfo containerChecksumInfo;
        block9: {
            long containerID = data.getContainerID();
            File checksumFile = ContainerChecksumTreeManager.getContainerChecksumFile(data);
            if (!checksumFile.exists()) {
                LOG.debug("No checksum file currently exists for container {} at the path {}", (Object)containerID, (Object)checksumFile);
                return ContainerProtos.ContainerChecksumInfo.newBuilder().build();
            }
            InputStream inStream = Files.newInputStream(checksumFile.toPath(), new OpenOption[0]);
            try {
                containerChecksumInfo = ContainerProtos.ContainerChecksumInfo.parseFrom((InputStream)inStream);
                if (inStream == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (inStream != null) {
                        try {
                            inStream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException ex) {
                    throw new IOException("Error occurred when reading container merkle tree for containerID " + data.getContainerID() + " at path " + checksumFile, ex);
                }
            }
            inStream.close();
        }
        return containerChecksumInfo;
    }

    @VisibleForTesting
    public ContainerMerkleTreeMetrics getMetrics() {
        return this.metrics;
    }
}

