/*
 * Decompiled with CFR 0.152.
 */
package org.apache.polaris.core.persistence.cache;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.RemovalListener;
import jakarta.annotation.Nonnull;
import jakarta.annotation.Nullable;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.polaris.core.PolarisCallContext;
import org.apache.polaris.core.PolarisDiagnostics;
import org.apache.polaris.core.config.BehaviorChangeConfiguration;
import org.apache.polaris.core.config.FeatureConfiguration;
import org.apache.polaris.core.config.RealmConfig;
import org.apache.polaris.core.entity.PolarisBaseEntity;
import org.apache.polaris.core.entity.PolarisChangeTrackingVersions;
import org.apache.polaris.core.entity.PolarisEntityId;
import org.apache.polaris.core.entity.PolarisEntityType;
import org.apache.polaris.core.entity.PolarisGrantRecord;
import org.apache.polaris.core.persistence.PolarisMetaStoreManager;
import org.apache.polaris.core.persistence.ResolvedPolarisEntity;
import org.apache.polaris.core.persistence.cache.EntityCache;
import org.apache.polaris.core.persistence.cache.EntityCacheByNameKey;
import org.apache.polaris.core.persistence.cache.EntityCacheLookupResult;
import org.apache.polaris.core.persistence.cache.EntityWeigher;
import org.apache.polaris.core.persistence.dao.entity.ChangeTrackingResult;
import org.apache.polaris.core.persistence.dao.entity.ResolvedEntitiesResult;
import org.apache.polaris.core.persistence.dao.entity.ResolvedEntityResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InMemoryEntityCache
implements EntityCache {
    private static final Logger LOGGER = LoggerFactory.getLogger(InMemoryEntityCache.class);
    public static final int MAX_CACHE_REFRESH_ATTEMPTS = 100;
    private final PolarisDiagnostics diagnostics;
    private final PolarisMetaStoreManager polarisMetaStoreManager;
    private final Cache<Long, ResolvedPolarisEntity> byId;
    private final AbstractMap<EntityCacheByNameKey, ResolvedPolarisEntity> byName;

    public InMemoryEntityCache(@Nonnull PolarisDiagnostics diagnostics, @Nonnull RealmConfig realmConfig, @Nonnull PolarisMetaStoreManager polarisMetaStoreManager) {
        this.diagnostics = diagnostics;
        this.byName = new ConcurrentHashMap<EntityCacheByNameKey, ResolvedPolarisEntity>();
        RemovalListener removalListener = (key, value, cause) -> {
            if (value != null) {
                EntityCacheByNameKey nameKey = new EntityCacheByNameKey(value.getEntity());
                this.byName.remove(nameKey, value);
            }
        };
        long weigherTarget = realmConfig.getConfig(FeatureConfiguration.ENTITY_CACHE_WEIGHER_TARGET);
        Caffeine byIdBuilder = Caffeine.newBuilder().maximumWeight(weigherTarget).weigher(EntityWeigher.asWeigher()).expireAfterAccess(1L, TimeUnit.HOURS).removalListener(removalListener);
        boolean useSoftValues = realmConfig.getConfig(BehaviorChangeConfiguration.ENTITY_CACHE_SOFT_VALUES);
        if (useSoftValues) {
            byIdBuilder.softValues();
        }
        this.byId = byIdBuilder.build();
        this.polarisMetaStoreManager = polarisMetaStoreManager;
    }

    @Override
    public void removeCacheEntry(@Nonnull ResolvedPolarisEntity cacheEntry) {
        EntityCacheByNameKey nameKey = new EntityCacheByNameKey(cacheEntry.getEntity());
        this.byId.asMap().remove(cacheEntry.getEntity().getId(), cacheEntry);
        this.byName.remove(nameKey, cacheEntry);
    }

    private void cacheNewEntry(@Nonnull ResolvedPolarisEntity cacheEntry) {
        EntityCacheByNameKey oldNameKey;
        EntityCacheByNameKey nameKey = new EntityCacheByNameKey(cacheEntry.getEntity());
        ResolvedPolarisEntity oldCacheEntry = (ResolvedPolarisEntity)this.byId.getIfPresent((Object)cacheEntry.getEntity().getId());
        this.byId.asMap().merge(cacheEntry.getEntity().getId(), cacheEntry, (oldValue, newValue) -> this.isNewer((ResolvedPolarisEntity)newValue, (ResolvedPolarisEntity)oldValue) ? newValue : oldValue);
        if (!cacheEntry.getEntity().isDropped()) {
            this.byName.put(nameKey, cacheEntry);
        }
        if (oldCacheEntry != null && !(oldNameKey = new EntityCacheByNameKey(oldCacheEntry.getEntity())).equals(nameKey)) {
            this.byName.remove(oldNameKey, oldCacheEntry);
        }
    }

    private boolean isNewer(ResolvedPolarisEntity newValue, ResolvedPolarisEntity oldValue) {
        return newValue.getEntity().getEntityVersion() > oldValue.getEntity().getEntityVersion() || newValue.getEntity().getGrantRecordsVersion() > oldValue.getEntity().getGrantRecordsVersion();
    }

    private void replaceCacheEntry(@Nullable ResolvedPolarisEntity oldCacheEntry, @Nonnull ResolvedPolarisEntity newCacheEntry) {
        if (oldCacheEntry != null) {
            if (this.entityNameKeyMismatch(oldCacheEntry.getEntity(), newCacheEntry.getEntity()) || oldCacheEntry.getEntity().getEntityVersion() < newCacheEntry.getEntity().getEntityVersion() || oldCacheEntry.getEntity().getGrantRecordsVersion() < newCacheEntry.getEntity().getGrantRecordsVersion()) {
                this.cacheNewEntry(newCacheEntry);
                this.removeCacheEntry(oldCacheEntry);
            }
        } else {
            this.cacheNewEntry(newCacheEntry);
        }
    }

    private boolean entityNameKeyMismatch(@Nonnull PolarisBaseEntity entity, @Nonnull PolarisBaseEntity otherEntity) {
        return entity.getId() != otherEntity.getId() || entity.getParentId() != otherEntity.getParentId() || !entity.getName().equals(otherEntity.getName()) || entity.getTypeCode() != otherEntity.getTypeCode();
    }

    @Nullable
    public ResolvedPolarisEntity getEntityById(long entityId) {
        return (ResolvedPolarisEntity)this.byId.getIfPresent((Object)entityId);
    }

    @Nullable
    public ResolvedPolarisEntity getEntityByName(@Nonnull EntityCacheByNameKey entityNameKey) {
        return this.byName.get(entityNameKey);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    @Nullable
    public ResolvedPolarisEntity getAndRefreshIfNeeded(@Nonnull PolarisCallContext callContext, @Nonnull PolarisBaseEntity entityToValidate, int entityMinVersion, int entityGrantRecordsMinVersion) {
        int grantRecordsVersion;
        List<PolarisGrantRecord> grantRecords;
        PolarisBaseEntity entity;
        long entityCatalogId = entityToValidate.getCatalogId();
        long entityId = entityToValidate.getId();
        PolarisEntityType entityType = entityToValidate.getType();
        ResolvedPolarisEntity existingCacheEntry = this.getEntityById(entityId);
        EntityCacheByNameKey nameKey = new EntityCacheByNameKey(entityToValidate);
        ResolvedPolarisEntity existingCacheEntryByName = this.getEntityByName(nameKey);
        if (existingCacheEntryByName != null && existingCacheEntry != null && this.isNewer(existingCacheEntry, existingCacheEntryByName)) {
            existingCacheEntry = existingCacheEntryByName;
        }
        if (existingCacheEntry != null && existingCacheEntry.getEntity().getEntityVersion() >= entityMinVersion) {
            if (existingCacheEntry.getEntity().getGrantRecordsVersion() >= entityGrantRecordsMinVersion) return existingCacheEntry;
        }
        if (existingCacheEntry == null) {
            ResolvedEntityResult refreshedCacheEntry = this.polarisMetaStoreManager.loadResolvedEntityById(callContext, entityCatalogId, entityId, entityType);
            if (!refreshedCacheEntry.isSuccess()) return null;
            entity = refreshedCacheEntry.getEntity();
            grantRecords = refreshedCacheEntry.getEntityGrantRecords();
            grantRecordsVersion = refreshedCacheEntry.getGrantRecordsVersion();
        } else {
            ResolvedEntityResult refreshedCacheEntry = this.polarisMetaStoreManager.refreshResolvedEntity(callContext, existingCacheEntry.getEntity().getEntityVersion(), existingCacheEntry.getEntity().getGrantRecordsVersion(), entityType, entityCatalogId, entityId);
            if (refreshedCacheEntry.isSuccess()) {
                PolarisBaseEntity polarisBaseEntity = entity = refreshedCacheEntry.getEntity() != null ? refreshedCacheEntry.getEntity() : existingCacheEntry.getEntity();
                if (refreshedCacheEntry.getEntityGrantRecords() != null) {
                    grantRecords = refreshedCacheEntry.getEntityGrantRecords();
                    grantRecordsVersion = refreshedCacheEntry.getGrantRecordsVersion();
                } else {
                    grantRecords = existingCacheEntry.getAllGrantRecords();
                    grantRecordsVersion = existingCacheEntry.getEntity().getGrantRecordsVersion();
                }
            } else {
                this.removeCacheEntry(existingCacheEntry);
                return null;
            }
        }
        this.diagnostics.checkNotNull(entity, "unexpected_null_entity");
        this.diagnostics.checkNotNull(grantRecords, "unexpected_null_grant_records");
        this.diagnostics.check(grantRecordsVersion > 0, "unexpected_null_grant_records_version");
        ResolvedPolarisEntity newCacheEntry = new ResolvedPolarisEntity(this.diagnostics, entity, grantRecords, grantRecordsVersion);
        this.replaceCacheEntry(existingCacheEntry, newCacheEntry);
        return newCacheEntry;
    }

    @Override
    @Nullable
    public EntityCacheLookupResult getOrLoadEntityById(@Nonnull PolarisCallContext callContext, long entityCatalogId, long entityId, PolarisEntityType entityType) {
        boolean cacheHit;
        ResolvedPolarisEntity entry = this.getEntityById(entityId);
        if (entry == null) {
            cacheHit = false;
            ResolvedEntityResult result = this.polarisMetaStoreManager.loadResolvedEntityById(callContext, entityCatalogId, entityId, entityType);
            if (!result.isSuccess()) {
                return null;
            }
            this.diagnostics.checkNotNull(result.getEntity(), "entity_should_loaded");
            this.diagnostics.checkNotNull(result.getEntityGrantRecords(), "entity_grant_records_should_loaded");
            entry = new ResolvedPolarisEntity(this.diagnostics, result.getEntity(), result.getEntityGrantRecords(), result.getGrantRecordsVersion());
            this.cacheNewEntry(entry);
        } else {
            cacheHit = true;
        }
        return new EntityCacheLookupResult(entry, cacheHit);
    }

    @Override
    @Nullable
    public EntityCacheLookupResult getOrLoadEntityByName(@Nonnull PolarisCallContext callContext, @Nonnull EntityCacheByNameKey entityNameKey) {
        boolean cacheHit;
        ResolvedPolarisEntity entry = this.getEntityByName(entityNameKey);
        if (entry == null) {
            cacheHit = false;
            ResolvedEntityResult result = this.polarisMetaStoreManager.loadResolvedEntityByName(callContext, entityNameKey.getCatalogId(), entityNameKey.getParentId(), entityNameKey.getType(), entityNameKey.getName());
            if (!result.isSuccess()) {
                return null;
            }
            this.diagnostics.checkNotNull(result.getEntity(), "entity_should_loaded");
            this.diagnostics.checkNotNull(result.getEntityGrantRecords(), "entity_grant_records_should_loaded");
            entry = new ResolvedPolarisEntity(this.diagnostics, result.getEntity(), result.getEntityGrantRecords(), result.getGrantRecordsVersion());
            this.cacheNewEntry(entry);
        } else {
            cacheHit = true;
        }
        return new EntityCacheLookupResult(entry, cacheHit);
    }

    @Override
    public List<EntityCacheLookupResult> getOrLoadResolvedEntities(@Nonnull PolarisCallContext callCtx, @Nonnull PolarisEntityType entityType, @Nonnull List<PolarisEntityId> entityIds) {
        HashMap<PolarisEntityId, ResolvedPolarisEntity> resolvedEntities = new HashMap<PolarisEntityId, ResolvedPolarisEntity>();
        boolean stateResolved = false;
        for (int i = 0; i < 100; ++i) {
            Function<List<PolarisEntityId>, ResolvedEntitiesResult> loaderFunc = idsToLoad -> this.polarisMetaStoreManager.loadResolvedEntities(callCtx, entityType, (List<PolarisEntityId>)idsToLoad);
            if (!this.isCacheStateValid(callCtx, resolvedEntities, entityIds, loaderFunc)) continue;
            stateResolved = true;
            break;
        }
        if (!stateResolved) {
            LOGGER.warn("Unable to resolve entities in cache after multiple attempts {} - resolved: {}", entityIds, resolvedEntities);
            this.diagnostics.fail("cannot_resolve_all_entities", "Unable to resolve entities in cache", new Object[0]);
        }
        return entityIds.stream().map(id -> {
            ResolvedPolarisEntity entity = (ResolvedPolarisEntity)resolvedEntities.get(id);
            return entity == null ? null : new EntityCacheLookupResult(entity, true);
        }).collect(Collectors.toList());
    }

    private boolean isCacheStateValid(@Nonnull PolarisCallContext callCtx, @Nonnull Map<PolarisEntityId, ResolvedPolarisEntity> resolvedEntities, @Nonnull List<PolarisEntityId> entityIds, @Nonnull Function<List<PolarisEntityId>, ResolvedEntitiesResult> loaderFunc) {
        ResolvedEntitiesResult resolvedEntitiesResult;
        ChangeTrackingResult changeTrackingResult = this.polarisMetaStoreManager.loadEntitiesChangeTracking(callCtx, entityIds);
        ArrayList<PolarisEntityId> idsToLoad = new ArrayList<PolarisEntityId>();
        if (changeTrackingResult.isSuccess()) {
            idsToLoad.addAll(this.validateCacheEntries(entityIds, resolvedEntities, changeTrackingResult));
        } else {
            idsToLoad.addAll(entityIds);
        }
        if (!idsToLoad.isEmpty() && (resolvedEntitiesResult = loaderFunc.apply(idsToLoad)).isSuccess()) {
            LOGGER.debug("Resolved entities - validating cache");
            resolvedEntitiesResult.getResolvedEntities().stream().filter(Objects::nonNull).forEach(e -> {
                this.cacheNewEntry((ResolvedPolarisEntity)e);
                resolvedEntities.put(new PolarisEntityId(e.getEntity().getCatalogId(), e.getEntity().getId()), (ResolvedPolarisEntity)e);
            });
        }
        List idsToReload = changeTrackingResult.isSuccess() ? this.validateCacheEntries(entityIds, resolvedEntities, changeTrackingResult) : List.of();
        return idsToReload.isEmpty();
    }

    private List<PolarisEntityId> validateCacheEntries(List<PolarisEntityId> entityIds, Map<PolarisEntityId, ResolvedPolarisEntity> resolvedEntities, ChangeTrackingResult changeTrackingResult) {
        ArrayList<PolarisEntityId> idsToReload = new ArrayList<PolarisEntityId>();
        Iterator<PolarisEntityId> idIterator = entityIds.iterator();
        Iterator<PolarisChangeTrackingVersions> changeTrackingIterator = changeTrackingResult.getChangeTrackingVersions().iterator();
        while (idIterator.hasNext() && changeTrackingIterator.hasNext()) {
            ResolvedPolarisEntity cachedEntity;
            PolarisEntityId entityId = idIterator.next();
            PolarisChangeTrackingVersions changeTrackingVersions = changeTrackingIterator.next();
            if (changeTrackingVersions == null) {
                cachedEntity = this.getEntityById(entityId.getId());
                if (cachedEntity == null && !resolvedEntities.containsKey(entityId)) continue;
                LOGGER.debug("Entity {} has been purged, removing from cache", (Object)entityId);
                Optional.ofNullable(cachedEntity).ifPresent(this::removeCacheEntry);
                resolvedEntities.remove(entityId);
                continue;
            }
            cachedEntity = resolvedEntities.computeIfAbsent(entityId, id -> this.getEntityById(id.getId()));
            if (cachedEntity == null || cachedEntity.getEntity().getEntityVersion() != changeTrackingVersions.getEntityVersion() || cachedEntity.getEntity().getGrantRecordsVersion() != changeTrackingVersions.getGrantRecordsVersion()) {
                idsToReload.add(entityId);
                continue;
            }
            resolvedEntities.put(entityId, cachedEntity);
        }
        LOGGER.debug("Cache entries {} need to be reloaded", idsToReload);
        return idsToReload;
    }
}

