/* The nasty work of reading 32 and 64-bit modules is in here. */
#include <elf.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include "depmod.h"
#include "moduleops.h"
#include "tables.h"

/* Load the given section: NULL on error. */
static void *load_section32(Elf32_Ehdr *hdr,
			    const char *secname,
			    unsigned long *size)
{
	Elf32_Shdr *sechdrs;
	unsigned int i;
	char *secnames;

	/* Grab section headers and strings so we can tell who is who */
	sechdrs = (void *)hdr + hdr->e_shoff;
	secnames = (void *)hdr + sechdrs[hdr->e_shstrndx].sh_offset;

	/* Find the section they want */
	for (i = 1; i < hdr->e_shnum; i++) {
		if (strcmp(secnames+sechdrs[i].sh_name, secname) == 0) {
			*size = sechdrs[i].sh_size;
			return (void *)hdr + sechdrs[i].sh_offset;
		}
	}
	*size = 0;
	return NULL;
}

static void load_symbols32(struct module *module)
{
	struct kernel_symbol32 *ksyms;
	unsigned long i, size;

	ksyms = load_section32(module->mmap, "__ksymtab", &size);
	for (i = 0; i < size / sizeof(struct kernel_symbol32); i++)
		add_symbol(ksyms[i].name, module);
}

/* Calculate the dependencies for this module */
static void calculate_deps32(struct module *module)
{
	unsigned int i;
	unsigned long size;
	char *strings;
	Elf32_Sym *syms;

	strings = load_section32(module->mmap, ".strtab", &size);
	syms = load_section32(module->mmap, ".symtab", &size);

	if (!strings || !syms) {
		warn("Couldn't find symtab and strtab in module %s\n",
		     module->pathname);
		return;
	}

	module->num_deps = 0;
	module->deps = NULL;
	for (i = 0; i < size / sizeof(syms[0]); i++) {
		if (syms[i].st_shndx == SHN_UNDEF) {
			/* Look for symbol */
			const char *name = strings + syms[i].st_name;
			struct module *owner;

			if (strcmp(name, "") == 0)
				continue;

			owner = find_symbol(name);
			if (owner)
				add_dep(module, owner);
			else
				unknown_symbol(module, name);
		}
	}
}

static void *deref_sym32(Elf32_Ehdr *hdr, const char *name)
{
	unsigned int i;
	unsigned long size;
	char *strings;
	Elf32_Sym *syms;
	Elf32_Shdr *sechdrs;

	sechdrs = (void *)hdr + hdr->e_shoff;
	strings = load_section32(hdr, ".strtab", &size);
	syms = load_section32(hdr, ".symtab", &size);

	/* Don't warn again: we already have above */
	if (!strings || !syms)
		return NULL;

	for (i = 0; i < size / sizeof(syms[0]); i++) {
		if (strcmp(strings + syms[i].st_name, name) == 0) {
			return (void *)hdr
				+ sechdrs[syms[i].st_shndx].sh_offset
				+ syms[i].st_value;
		}
	}
	return NULL;
}

/* FIXME: Check size, unless we end up using aliases anyway --RR */
static void fetch_tables32(struct module *module)
{
	module->pci_size = PCI_DEVICE_SIZE32;
	module->pci_table = deref_sym32(module->mmap,
					"__mod_pci_device_table");

	module->usb_size = USB_DEVICE_SIZE32;
	module->usb_table = deref_sym32(module->mmap,
					"__mod_usb_device_table");
}

struct module_ops mod32_ops = {
	.load_symbols	= load_symbols32,
	.calculate_deps	= calculate_deps32,
	.fetch_tables	= fetch_tables32,
};

/* Load the given section: NULL on error. */
static void *load_section64(Elf64_Ehdr *hdr,
			    const char *secname,
			    unsigned long *size)
{
	Elf64_Shdr *sechdrs;
	unsigned int i;
	char *secnames;

	/* Grab section headers and strings so we can tell who is who */
	sechdrs = (void *)hdr + hdr->e_shoff;
	secnames = (void *)hdr + sechdrs[hdr->e_shstrndx].sh_offset;

	/* Find the section they want */
	for (i = 1; i < hdr->e_shnum; i++) {
		if (strcmp(secnames+sechdrs[i].sh_name, secname) == 0) {
			*size = sechdrs[i].sh_size;
			return (void *)hdr + sechdrs[i].sh_offset;
		}
	}
	*size = 0;
	return NULL;
}

static void load_symbols64(struct module *module)
{
	struct kernel_symbol64 *ksyms;
	unsigned long i, size;

	ksyms = load_section64(module->mmap, "__ksymtab", &size);
	for (i = 0; i < size / sizeof(struct kernel_symbol64); i++)
		add_symbol(ksyms[i].name, module);
}

/* Calculate the dependencies for this module */
static void calculate_deps64(struct module *module)
{
	unsigned int i;
	unsigned long size;
	char *strings;
	Elf64_Sym *syms;

	strings = load_section64(module->mmap, ".strtab", &size);
	syms = load_section64(module->mmap, ".symtab", &size);

	if (!strings || !syms) {
		warn("Couldn't find symtab and strtab in module %s\n",
		     module->pathname);
		return;
	}

	module->num_deps = 0;
	module->deps = NULL;
	for (i = 0; i < size / sizeof(syms[0]); i++) {
		if (syms[i].st_shndx == SHN_UNDEF) {
			/* Look for symbol */
			const char *name = strings + syms[i].st_name;
			struct module *owner;

			if (strcmp(name, "") == 0)
				continue;

			owner = find_symbol(name);
			if (owner)
				add_dep(module, owner);
			else
				unknown_symbol(module, name);
		}
	}
}

static void *deref_sym64(Elf64_Ehdr *hdr, const char *name)
{
	unsigned int i;
	unsigned long size;
	char *strings;
	Elf64_Sym *syms;
	Elf64_Shdr *sechdrs;

	sechdrs = (void *)hdr + hdr->e_shoff;
	strings = load_section64(hdr, ".strtab", &size);
	syms = load_section64(hdr, ".symtab", &size);

	/* Don't warn again: we already have above */
	if (!strings || !syms)
		return NULL;

	for (i = 0; i < size / sizeof(syms[0]); i++) {
		if (strcmp(strings + syms[i].st_name, name) == 0)
			return (void *)hdr 
				+ sechdrs[syms[i].st_shndx].sh_offset
				+ syms[i].st_value;
	}
	return NULL;
}

/* FIXME: Check size, unless we end up using aliases anyway --RR */
static void fetch_tables64(struct module *module)
{
	module->pci_size = PCI_DEVICE_SIZE64;
	module->pci_table = deref_sym64(module->mmap,
					"__module_pci_device_table");

	module->usb_size = USB_DEVICE_SIZE64;
	module->usb_table = deref_sym64(module->mmap,
					"__module_usb_device_table");
}

struct module_ops mod64_ops = {
	.load_symbols	= load_symbols64,
	.calculate_deps	= calculate_deps64,
	.fetch_tables	= fetch_tables64,
};
