/* modprobe.c: insert a module into the kernel, intelligently.
    Copyright (C) 2001  Rusty Russell.
    Copyright (C) 2002  Rusty Russell, IBM Corporation.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <sys/utsname.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <dirent.h>
#include <limits.h>
#include <elf.h>
#include <getopt.h>
#include <fnmatch.h>
#include <asm/unistd.h>
#include <sys/wait.h>

#include "backwards_compat.c"

struct module_list;

struct module {
	struct module		*next;
	struct module_list	*dependencies;
	enum { NOT_LOADED = 0, BEING_LOADED, LOADED } state;
	char			filename[0];
};

struct module_list {
	struct module		*module;
	struct module_list	*next;
};

#define MODULE_DIR "/lib/modules"

static void fatal(const char *fmt, ...)
__attribute__ ((noreturn, format (printf, 1, 2)));

static void fatal(const char *fmt, ...)
{
	va_list arglist;

	fprintf(stderr, "FATAL: ");

	va_start(arglist, fmt);
	vfprintf(stderr, fmt, arglist);
	va_end(arglist);

	exit(1);
}

static void warn(const char *fmt, ...)
__attribute__ ((format (printf, 1, 2)));

static void warn(const char *fmt, ...)
{
	va_list arglist;

	fprintf(stderr, "WARNING: ");

	va_start(arglist, fmt);
	vfprintf(stderr, fmt, arglist);
	va_end(arglist);
}

static void *do_nofail(void *ptr, const char *file, int line, const char *expr)
{
	if (!ptr) {
		fatal("Memory allocation failure %s line %d: %s.\n",
		      file, line, expr);
	}
	return ptr;
}

#define NOFAIL(ptr)	do_nofail((ptr), __FILE__, __LINE__, #ptr)

static void print_usage(const char *progname)
{
	fprintf(stderr,
		"Usage: %s [--verbose|--version|--config] filename options\n",
		progname);
	exit(1);
}

static int fgetc_wrapped(FILE *file)
{
	for (;;) {
	  	int ch = fgetc(file);
		if (ch != '\\')
			return ch;
		ch = fgetc(file);
		if (ch != '\n')
			return ch;
	}
}

static char *getline_wrapped(FILE *file)
{
	int size = 1024;
	int i = 0;
	char *buf = NOFAIL(malloc(size));
	for(;;) {
		int ch = fgetc_wrapped(file);
		if (i == size) {
			size *= 2;
			buf = NOFAIL(realloc(buf, size));
		}
		if (ch < 0 || ch == '\n') {
			if (ch < 0 && i == 0) {
				free(buf);
				return NULL;
			}
			buf[i] = '\0';
			return NOFAIL(realloc(buf, i+1));
		}
		buf[i++] = ch;
	}
}

struct module *modules;		/* = NULL */

struct module *get_module(char *filename, int namelen)
{
	struct module *mod;
	for (mod = modules; mod; mod = mod->next) {
		if (strlen(mod->filename) == namelen &&
		    memcmp(mod->filename, filename, namelen) == 0)
			return mod;
	}

	/* No match.  Make a new module. */
	mod = NOFAIL(malloc(sizeof(struct module) + namelen + 1));
	memset(mod, 0, sizeof(struct module));
	memcpy(mod->filename, filename, namelen);
	mod->filename[namelen] = '\0';
	mod->next = modules;
	modules = mod;
	return mod;
}

static void add_modules_dep_line(char *line, const char *start_name,
				 struct module **start)
{
	struct module_list *dep;
	char *dep_start;
	struct module *mod, *dep_mod;
	char *ptr;
	int len;
	char *modname;

	/* Ignore lines without : or which start with a # */
	ptr = index(line, ':');
	if (ptr == NULL || line[strspn(line, "\t ")] == '#')
		return;

	mod = get_module(line, ptr - line);

	modname = strrchr(mod->filename, '/') + 1;
	len = strlen(modname);
	if (strchr(modname, '.'))
		len = strchr(modname, '.') - modname;

	if (len == strlen(start_name) && !strncmp(modname, start_name, len))
		*start = mod;

	ptr++;
	for(;;) {
		ptr += strspn(ptr, " \t");
		if (*ptr == '\0')
			break;
		dep_start = ptr;
		ptr += strcspn(ptr, " \t");
		dep_mod = get_module(dep_start, ptr - dep_start);
		fflush(stdout);
		dep = NOFAIL(malloc(sizeof(*dep)));
		dep->module = dep_mod;
		dep->next = mod->dependencies;
		mod->dependencies = dep;
	}
}

static struct module *read_depends(const char *dirname,
				   const char *start_name)
{
	struct module *start;
	char modules_dep_name[strlen(dirname) + sizeof("modules.dep") + 1];
	char *line;
	FILE *modules_dep;

	start = NULL;
	sprintf(modules_dep_name, "%s/%s", dirname, "modules.dep");
	modules_dep = fopen(modules_dep_name, "r");
	if (!modules_dep)
		fatal("Could not load %s: %s\n",
		      modules_dep_name, strerror(errno));

	while((line = getline_wrapped(modules_dep)) != NULL) {
		add_modules_dep_line(line, start_name, &start);
		free(line);
	}

	fclose(modules_dep);
	return start;
}

/* We use error numbers in a loose translation... */
static const char *moderror(int err)
{
	switch (err) {
	case ENOEXEC:
		return "Invalid module format";
	case ENOENT:
		return "Unknown symbol in module";
	default:
		return strerror(err);
	}
}

static int
module_in_kernel(const char *modname)
{
	FILE *proc_modules;
	char *line;
	const int modname_len = strlen(modname);
	
	proc_modules = fopen("/proc/modules", "r");
	if (proc_modules == NULL)
		fatal("Error reading /proc/modules: %s\n", strerror(errno));

	while ((line = getline_wrapped(proc_modules)) != NULL) {
		if (strncmp(line, modname, modname_len) == 0 &&
		    isspace(line[modname_len])) {
			free(line);
			fclose(proc_modules);
			return 1;
		}
		free(line);
	}
	fclose(proc_modules);
	return 0;
}

static void *read_in(int fd, unsigned long len)
{
	void *ret = NOFAIL(malloc(len));
	unsigned long done = 0;
	int r;

	while (done < len) {
		r = read(fd, ret + done, len - done);
		if (r <= 0) {
			free(ret);
			return NULL;
		}
		done += r;
	}
	return ret;
}

static void rename_module32(struct module *module,
			    void *mod,
			    unsigned long len,
			    const char *newname)
{
	Elf32_Ehdr *hdr = mod;
	Elf32_Shdr *sechdrs = mod + hdr->e_shoff;
	const char *secnames = mod + sechdrs[hdr->e_shstrndx].sh_offset;
	unsigned int i;

	/* Replace the .gnu.linkonce.modname section */
	for (i = 1; i < hdr->e_shnum; i++) {
		if (strcmp(secnames+sechdrs[i].sh_name,
			   ".gnu.linkonce.modname") == 0) {
			strncpy(mod + sechdrs[i].sh_offset, newname,
				sechdrs[i].sh_size - 1);
			return;
		}
	}
	warn("Could not find module name to change in %s\n", module->filename);
}

static void rename_module64(struct module *module,
			  void *mod,
			  unsigned long len,
			  const char *newname)
{
	Elf64_Ehdr *hdr = mod;
	Elf64_Shdr *sechdrs = mod + hdr->e_shoff;
	const char *secnames = mod + sechdrs[hdr->e_shstrndx].sh_offset;
	unsigned int i;

	/* Replace the .gnu.linkonce.modname section */
	for (i = 1; i < hdr->e_shnum; i++) {
		if (strcmp(secnames+sechdrs[i].sh_name,
			   ".gnu.linkonce.modname") == 0) {
			strncpy(mod + sechdrs[i].sh_offset, newname,
				sechdrs[i].sh_size - 1);
			return;
		}
	}
	warn("Could not find module name to change in %s\n", module->filename);
}

static void rename_module(struct module *module,
			  void *mod,
			  unsigned long len,
			  const char *newname)
{
	/* "\177ELF" <byte> where byte = 001 for 32-bit, 002 for 64 */
	char *ident = mod;

	if (memcmp(mod, ELFMAG, SELFMAG) != 0) {
		warn("Unknown module format in %s: not changing name\n",
		     module->filename);
		return;
	}

	switch (ident[EI_CLASS]) {
	case ELFCLASS32:
		rename_module32(module, mod, len, newname);
		break;
	case ELFCLASS64:
		rename_module64(module, mod, len, newname);
		break;
	default:
		warn("Module %s has elf unknown identifier %i\n",
		     module->filename, ident[EI_CLASS]);
	}
}

struct module_options
{
	struct module_options *next;
	char *modulename;
	char *options;
};

struct module_installcmd
{
	struct module_installcmd *next;
	char *modulename;
	char *installcmd;
};

/* Link in a new option line from the config file. */
static struct module_options *
add_options(const char *modname,
	    const char *option,
	    struct module_options *options)
{
	struct module_options *new;

	new = NOFAIL(malloc(sizeof(*new)));
	new->modulename = NOFAIL(strdup(modname));
	new->options = NOFAIL(strdup(option));
	new->next = options;
	return new;
}

/* Link in a new install line from the config file. */
static struct module_installcmd *
add_installcmd(const char *modname,
	       const char *installcmd,
	       struct module_installcmd *installcmds)
{
	struct module_installcmd *new;

	new = NOFAIL(malloc(sizeof(*new)));
	new->modulename = NOFAIL(strdup(modname));
	new->installcmd = NOFAIL(strdup(installcmd));
	new->next = installcmds;
	return new;
}

/* Find install commands if any. */
static const char *find_installcmd(const char *modname,
				   const struct module_installcmd *installcmds)
{
	while (installcmds) {
		if (strcmp(installcmds->modulename, modname) == 0)
			return installcmds->installcmd;
		installcmds = installcmds->next;
	}
	return NULL;
}

/* Add to options */
static char *add_extra_options(const char *modname,
			       const char *base_options,
			       const struct module_options *options)
{
	char *opts = NOFAIL(strdup(base_options));
	
	while (options) {
		if (strcmp(options->modulename, modname) == 0) {
			opts = NOFAIL(realloc(opts,
					      strlen(opts) + 1
					      + strlen(options->options) + 1));
			strcat(opts, " ");
			strcat(opts, options->options);
		}
		options = options->next;
	}
	return opts;
}

/* Do an install command. */
static void do_installcmd(const char *modname,
			  const char *installcmd,
			  int verbose, int dry_run, int dont_fail)
{
	int ret;

	if (verbose)
		printf("install %s: %s\n", modname, installcmd);
	if (dry_run)
		return;

	ret = system(installcmd);
	if (ret == -1 || WEXITSTATUS(ret)) {
		if (dont_fail)
			fatal("Error running install command for %s\n",
			      modname);
		warn("Error running install command for %s\n",modname);
	}
}

/* Actually do the insert. */
static void insmod(struct module *mod,
		   const char *base_options,
		   const char *newname,
		   int dont_fail,
		   int dry_run,
		   int verbose,
		   const struct module_options *options,
		   const struct module_installcmd *installcmds,
		   struct module_list *caller)
{
	int fd, ret;
	struct stat st;
	void *map;
	struct module_list *dep;
	unsigned int i;
	const char *installcmd;
	char *optstring;
	char modname[strlen(strrchr(mod->filename, '/') ?: mod->filename)];
	struct module_list call_history, *modlist;

	if (mod->state == LOADED)
		return;

	call_history.next = caller;
	call_history.module = mod;

	if (mod->state == BEING_LOADED) {
		fprintf (stderr, "FATAL: recursive dependency loop:\n");
		modlist = &call_history;
		do {
			fprintf(stderr, "\t%s\n", modlist->module->filename);
			modlist = modlist->next;
		} while (modlist != NULL);
		exit(1);
	}

	mod->state = BEING_LOADED;

	for (dep = mod->dependencies; dep != NULL; dep = dep->next)
		insmod(dep->module, "", NULL, 0, dry_run, verbose,
		       options, installcmds, &call_history);

	/* If we fail to load after this point, we abort the whole program. */
	mod->state = LOADED;

	/* Now, it may already be loaded: check /proc/modules */
	strcpy(modname, strrchr(mod->filename, '/') + 1);

	/* Convert to underscores, truncate at last . */
	for (i = 0; modname[i]; i++)
		if (modname[i] == '-')
			modname[i] = '_';
	if (strrchr(modname, '.'))
		*strrchr(modname, '.') = '\0';

	if (module_in_kernel(newname ?: modname)) {
		if (dont_fail)
			fatal("Module %s already in kernel.\n",
			      newname ?: modname);
		return;
	}

	/* Open and suck in module (don't mmap, we might want to mangle) */
	fd = open(mod->filename, O_RDONLY);
	if (fd < 0) {
		if (dont_fail)
			fatal("Could not open `%s': %s\n",
			      mod->filename, strerror(errno));
		warn("Could not open `%s': %s\n",
		     mod->filename, strerror(errno));
		return;
	}

	fstat(fd, &st);
	map = read_in(fd, st.st_size);
	if (!map) {
		if (dont_fail)
			fatal("Can't read `%s': %s\n",
			      mod->filename, strerror(errno));
		warn("Can't read `%s': %s\n", mod->filename, strerror(errno));
		goto out_fd;
	}

	/* Rename it? */
	if (newname)
		rename_module(mod, map, st.st_size, newname);

	/* Did config file override installcmd or add options? */
	installcmd = find_installcmd(modname, installcmds);
	optstring = add_extra_options(modname, base_options, options);

	if (installcmd)
		do_installcmd(modname, installcmd,
			      verbose, dry_run, dont_fail);

	if (verbose)
		printf ("insmod %s %s\n", mod->filename, optstring);

	if (dry_run)
		goto out;

	ret = syscall(__NR_init_module, map, st.st_size, optstring);
	if (ret != 0) {
		if (dont_fail)
			fatal("Error inserting %s (%s): %s\n",
			      modname, mod->filename, moderror(errno));
		warn("Error inserting %s (%s): %s\n",
		     modname, mod->filename, moderror(errno));
	}
 out:
	free(map);
	free(optstring);
 out_fd:
	close(fd);
}

static char *strsep_skipspace(char **string, char *delim)
{
	*string += strspn(*string, delim);
	return strsep(string, delim);
}

/* FIXME: Maybe should be extended to "alias a b [and|or c]...".
   modules.dep must also contain aliases extracted from modules. --RR */

/* Simple format, ignore lines starting with #, one command per line.
   Returns NULL or resolved alias. */
static char *read_config(const char *filename,
			 int mustload,
			 char *name,
			 int dump_only,
			 struct module_options **options,
			 struct module_installcmd **installcmds)
{
	FILE *cfile;
	char *line;
	char *result = NULL;

	cfile = fopen(filename, "r");
	if (!cfile) {
		if (mustload)
			fatal("Failed to open config file %s: %s\n",
			      filename, strerror(errno));
		return NOFAIL(strdup(name));
	}

	while ((line = getline_wrapped(cfile)) != NULL) {
		char *ptr = line;
		char *cmd, *modname;

		if (dump_only)
			printf("%s\n", line);

		cmd = strsep_skipspace(&ptr, "\t ");
		if (cmd == NULL)
			continue;

		if (strcmp(cmd, "alias") == 0) {
			char *fakename = strsep_skipspace(&ptr, "\t ");
			char *realname = strsep_skipspace(&ptr, "\t ");

			if (fakename && realname && !strcmp(fakename, name))
				result = NOFAIL(strdup(realname));
		} else if (strcmp(cmd, "include") == 0) {
			filename = strsep(&ptr, "\t ");
			result = read_config(filename, 0, name, dump_only,
					     options, installcmds);
		} else if (strcmp(cmd, "options") == 0) {
			modname = strsep(&ptr, "\t ");
			*options = add_options(modname, ptr, *options);
		} else if (strcmp(cmd, "install") == 0) {
			modname = strsep(&ptr, "\t ");
			*installcmds =add_installcmd(modname,ptr,*installcmds);
		}
		free(line);
	}
	fclose(cfile);
	return result;
}

static struct option options[] = { { "verbose", 0, NULL, 'v' },
				   { "version", 0, NULL, 'V' },
				   { "config", 1, NULL, 'C' },
				   { "name", 1, NULL, 'o' },
				   { "showconfig", 0, NULL, 'c' },
				   { "autoclean", 0, NULL, 'k' },
				   { "quiet", 0, NULL, 'q' },
				   { "show", 0, NULL, 'n' },
				   { "syslog", 0, NULL, 's' },
				   { NULL, 0, NULL, 0 } };

#define DEFAULT_CONFIG "/etc/modprobe.conf"

int main(int argc, char *argv[])
{
	struct utsname buf;
	int opt;
	int dump_only = 0;
	int dry_run = 0;
	int verbose = 0;
	const char *config = NULL, *installcmd;
	char *dirname, *optstring = NOFAIL(strdup(""));
	char *modname, *newname = NULL;
	struct module_installcmd *installcmds = NULL;
	struct module_options *modoptions = NULL;
	struct module *start;

	try_old_version("modprobe", argv);

	while ((opt = getopt_long(argc, argv, "vVC:o:knqsc", options, NULL)) != -1){
		switch (opt) {
		case 'v':
			verbose = 1;
			break;
		case 'V':
			puts(PACKAGE " version " VERSION);
			exit(0);
		case 'C':
			config = optarg;
			break;
		case 'o':
			newname = optarg;
			break;
		case 'c':
			dump_only = 1;
			break;
		case 'k':
			/* FIXME: This should actually do something */
			break;
		case 'n':
			dry_run = 1;
			break;
		case 'q':
			/* FIXME: This should actually do something */
			break;
		case 's':
			/* FIXME: This should actually do something */
			break;
		default:
			print_usage(argv[0]);
		}
	}

	if (argc < optind + 1 && !dump_only)
		print_usage(argv[0]);

	/* Rest is module options */
	for (opt = optind + 1; opt < argc; opt++) {
		optstring = NOFAIL(realloc(optstring,
					   strlen(optstring) + 2
					   + strlen(argv[opt]) + 2));
		/* Spaces handled by "" pairs, but no way of escaping
                   quotes */
		if (strchr(argv[opt], ' '))
			strcat(optstring, "\"");
		strcat(optstring, argv[opt]);
		if (strchr(argv[opt], ' '))
			strcat(optstring, "\"");
		strcat(optstring, " ");
	}

	uname(&buf);
	dirname = NOFAIL(malloc(strlen(buf.release) + sizeof(MODULE_DIR) + 1));
	sprintf(dirname, "%s/%s", MODULE_DIR, buf.release);

	/* Returns the resolved alias, options */
	modname = read_config(config ?: DEFAULT_CONFIG,
			      config ? 1 : 0,
			      dump_only ? "" : argv[optind],
			      dump_only,
			      &modoptions, &installcmds);
	if (dump_only)
		exit(0);

	if (modname == NULL)
		modname = NOFAIL(strdup(argv[optind]));

	/* The dependencies have to be real modules, but the first one
           can be completely bogus (as long as there's an installcmd). */
	start = read_depends(dirname, modname);
	installcmd = find_installcmd(modname, installcmds);
	if (start) {
		insmod(start, optstring, newname, 1, dry_run, verbose,
		       modoptions, installcmds, NULL);
	} else {
		if (!installcmd)
			fatal("Module %s not found.\n", modname);
		do_installcmd(modname, installcmd, verbose, dry_run,1);
	}
	free(dirname);
	free(modname);

	return 0;
}
