/* New simplified depmod without backwards compat stuff and not
   requiring ksyms.

   Use the old one for the moment (with -F System.map).
   (C) 2002 Rusty Russell IBM Corporation
 */
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <elf.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <dirent.h>
#include <sys/utsname.h>
#include <sys/mman.h>

#include "depmod.h"
#include "moduleops.h"
#include "tables.h"

/* I hate strcmp. */
#define streq(a,b) (strcmp((a),(b)) == 0)

void fatal(const char *fmt, ...)
{
	va_list arglist;

	fprintf(stderr, "FATAL: ");

	va_start(arglist, fmt);
	vfprintf(stderr, fmt, arglist);
	va_end(arglist);

	exit(1);
}

void warn(const char *fmt, ...)
{
	va_list arglist;

	fprintf(stderr, "WARNING: ");

	va_start(arglist, fmt);
	vfprintf(stderr, fmt, arglist);
	va_end(arglist);
}

void *do_nofail(void *ptr, const char *file, int line, const char *expr)
{
	if (!ptr) {
		fatal("Memory allocation failure %s line %d: %s.\n",
		      file, line, expr);
	}
	return ptr;
}

#define SYMBOL_HASH_SIZE 1024
struct symbol
{
	struct symbol *next;
	struct module *owner;
	char name[0];
};

static struct symbol *symbolhash[SYMBOL_HASH_SIZE];

/* This is based on the hash agorithm from gdbm, via tdb */
static inline unsigned int tdb_hash(const char *name)
{
	unsigned value;	/* Used to compute the hash value.  */
	unsigned   i;	/* Used to cycle through random values. */

	/* Set the initial value from the key size. */
	for (value = 0x238F13AF * strlen(name), i=0; name[i]; i++)
		value = (value + (((unsigned char *)name)[i] << (i*5 % 24)));

	return (1103515243 * value + 12345);
}

void add_symbol(const char *name, struct module *owner)
{
	unsigned int hash;
	struct symbol *new = NOFAIL(malloc(sizeof *new + strlen(name) + 1));

	new->owner = owner;
	strcpy(new->name, name);

	hash = tdb_hash(name) % SYMBOL_HASH_SIZE;
	new->next = symbolhash[hash];
	symbolhash[hash] = new;
}

struct module *find_symbol(const char *name)
{
	struct symbol *s;

	for (s = symbolhash[tdb_hash(name) % SYMBOL_HASH_SIZE]; s; s=s->next) {
		if (streq(s->name, name))
			return s->owner;
	}

	return NULL;
}

void add_dep(struct module *mod, struct module *depends_on)
{
	unsigned int i;

	for (i = 0; i < mod->num_deps; i++)
		if (mod->deps[i] == depends_on)
			return;

	mod->deps = realloc(mod->deps, sizeof(mod->deps[0])*(mod->num_deps+1));
	mod->deps[mod->num_deps++] = depends_on;
}

/* FIXME: Implement -F System.map and -e and do something here. */
void unknown_symbol(struct module *mod, const char *symbol)
{
}

static struct option options[] = { { "all", 0, NULL, 'a' },
				   { "quick", 0, NULL, 'A' },
				   { "basedir", 0, NULL, 'b' },
				   { "errsyms", 0, NULL, 'e' },
				   { "filesyms", 0, NULL, 'F' },
				   { "help", 0, NULL, 'h' },
				   { "show", 0, NULL, 'n' },
				   { "quiet", 0, NULL, 'q' },
				   { "root", 0, NULL, 'r' },
				   { "unresolved-error", 0, NULL, 'u' },
				   { "verbose", 0, NULL, 'v' },
				   { "version", 0, NULL, 'V' },
				   { NULL, 0, NULL, 0 } };

/* Version number or module name?  Don't assume extension. */
static int is_version_number(const char *version)
{
	unsigned int dummy;

	return (sscanf(version, "%u.%u.%u", &dummy, &dummy, &dummy) == 3);
}

static int old_module_version(const char *version)
{
	/* Expect three part version. */
	unsigned int major, sub, minor;

	sscanf(version, "%u.%u.%u", &major, &sub, &minor);

	if (major > 2) return 0;
	if (major < 2) return 1;

	/* 2.x */
	if (sub > 5) return 0;
	if (sub < 5) return 1;

	/* 2.5.x */
	if (minor >= 47) return 0;
	return 1;
}

static void exec_old_depmod(char *argv[])
{
	char *sep;
	char pathname[strlen(argv[0])+1];
	char oldname[strlen("depmod") + strlen(argv[0]) + sizeof(".old")];

	memset(pathname, 0, strlen(argv[0])+1);
	sep = strrchr(argv[0], '/');
	if (sep)
		memcpy(pathname, argv[0], sep - argv[0]+1);
	sprintf(oldname, "%s%s.old", pathname, "depmod");

	/* Recursion detection: we need an env var since we can't
	   change argv[0] (as older modutils uses it to determine
	   behavior). */
	if (getenv("MODULE_RECURSE"))
		return;
	setenv("MODULE_RECURSE", "y", 0);

	execvp(oldname, argv);
	fprintf(stderr,
		"Version requires old depmod, but couldn't run %s: %s\n",
		oldname, strerror(errno));
	exit(2);
}

static void print_usage(const char *name)
{
	fprintf(stderr,
	"%s " VERSION " -- part of " PACKAGE "\n"
	"%s -[aA] [-n -e -v -q -V -r -u]\n"
	"      [-b basedirectory] [forced_version]\n"
	"depmod [-n -e -v -q -r -u] [-F kernelsyms] module1.o module2.o ...\n"
	"If no arguments (except options) are given, \"depmod -a\" is assumed\n"
	"\n"
	"depmod will output a dependancy list suitable for the modprobe utility.\n"
	"\n"
	"\n"
	"Options:\n"
	"\t-a, --all            Probe all modules\n"
	"\t-n, --show           Write the dependency file on stdout only\n"
	"\t-V, --version        Print the release version\n"
	"\t-h, --help           Print this usage message\n"
	"\n"
	"The following options are useful for people managing distributions:\n"
	"\t-b basedirectory\n"
	"\t    --basedir basedirectory    Use an image of a module tree.\n"
	"\t-F kernelsyms\n"
	"\t    --filesyms kernelsyms      Use the file instead of the\n"
	"\t                               current kernel symbols.\n",
	"depmod", "depmod");
}

static int ends_in(const char *name, const char *ext)
{
	unsigned int namelen, extlen;

	/* Grab lengths */
	namelen = strlen(name);
	extlen = strlen(ext);

	if (namelen < extlen) return 0;

	if (streq(name + namelen - extlen, ext))
		return 1;
	return 0;
}

static struct module *grab_module(const char *dirname,
				  const char *filename,
				  struct module *next)
{
	struct stat buf;
	struct module *new;
	int fd;

	new = NOFAIL(malloc(sizeof(*new)
			    + strlen(dirname) + 1 + strlen(filename) + 1));
	sprintf(new->pathname, "%s/%s", dirname, filename);

	fd = open(new->pathname, O_RDONLY);
	if (fd < 0) {
		warn("Can't read module %s: %s\n",
		     new->pathname, strerror(errno));
		goto fail;
	}
	fstat(fd, &buf);
	new->mmap = mmap(0, buf.st_size, PROT_READ, MAP_PRIVATE, fd, 0);
	if (new->mmap == MAP_FAILED) {
		warn("Can't map module %s: %s\n",
		     new->pathname, strerror(errno));
		goto fail;
	}

	/* "\177ELF" <byte> where byte = 001 for 32-bit, 002 for 64 */
	if (memcmp(new->mmap, ELFMAG, SELFMAG) != 0) {
		warn("Module %s is not an elf object\n", new->pathname);
		goto fail;
	}

	switch (((char *)new->mmap)[EI_CLASS]) {
	case ELFCLASS32:
		new->ops = &mod32_ops;
		break;
	case ELFCLASS64:
		new->ops = &mod64_ops;
		break;
	default:
		warn("Module %s has elf unknown identifier %i\n",
		     new->pathname, ((char *)new->mmap)[EI_CLASS]);
		goto fail;
	}
	new->ops->load_symbols(new);
	new->ops->fetch_tables(new);

	new->next = next;
	close(fd);
	return new;

 fail:
	munmap(new->mmap, buf.st_size);
	close(fd);
	free(new);
	return next;
}

static void write_dep(struct module *mod, unsigned int skipchars, FILE *out)
{
	unsigned int i;

	for (i = 0; i < mod->num_deps; i++) {
		fprintf(out, " %s", mod->deps[i]->pathname + skipchars);
		write_dep(mod->deps[i], skipchars, out);
	}
}

/* FIXME: Don't write same dep twice: order and loop detect. --RR */
static void output_deps(struct module *modules,
			unsigned int skipchars,
			FILE *out)
{
	struct module *i;

	for (i = modules; i; i = i->next)
		i->ops->calculate_deps(i);

	/* Now dump them out. */
	for (i = modules; i; i = i->next) {
		fprintf(out, "%s:", i->pathname + skipchars);
		write_dep(i, skipchars, out);
		fprintf(out, "\n");
	}
}

static int smells_like_module(const char *name)
{
	if (ends_in(name,".o") || ends_in(name,".klm") || ends_in(name,".ko"))
		return 1;
	return 0;
}

static struct module *grab_dir(const char *dirname, struct module *next)
{
	DIR *dir;
	struct dirent *dirent;

	dir = opendir(dirname);
	if (!dir) {
		warn("Couldn't open directory %s: %s\n",
		     dirname, strerror(errno));
		return next;
	}

	while ((dirent = readdir(dir)) != NULL) {
		/* Several competing extensions here.  .klm, .ko, .o */
		if (smells_like_module(dirent->d_name))
			next = grab_module(dirname, dirent->d_name, next);
		else if (!streq(dirent->d_name, ".")
			 && !streq(dirent->d_name, "..")) {
			struct stat st;

			char subdir[strlen(dirname) + 1
				   + strlen(dirent->d_name) + 1];
			sprintf(subdir, "%s/%s", dirname, dirent->d_name);
			if (stat(subdir, &st) != 0)
				warn("Couldn't stat %s: %s\n", subdir,
				     strerror(errno));
			else if (S_ISDIR(st.st_mode))
				next = grab_dir(subdir, next);
		}
	}
	return next;
}

int main(int argc, char *argv[])
{
	int opt, all = 0;
	unsigned int skipchars = 0;
	FILE *depout = NULL, *pciout, *usbout;
	char *basedir = "/lib/modules", *dirname, *version;
	struct module *list = NULL;

	while ((opt = getopt_long(argc, argv, "ab:AehnqruvVF:", options, NULL))
	       != -1) {
		switch (opt) {
		case 'a':
			all = 1;
			break;
		case 'b':
			basedir = optarg;
			skipchars = strlen(basedir);
			break;
		case 'A':
			/* FIXME: Implement this optimization. */
			break;
		case 'F':
		case 'e':
			/* FIXME: Implement these together */
			break;
		case 'u':
		case 'v':
		case 'q':
			/* Ignored. */
			break;
		case 'h':
			print_usage(argv[0]);
			break;
		case 'n':
			depout = pciout = usbout = stdout;
			break;
		case 'V':
			printf("%s %s\n", PACKAGE, VERSION);
			exit(0);
		default:
			fprintf(stderr, "Unknown option `%s'\n", argv[optind]);
			print_usage(argv[0]);
		}
	}

	/* They can specify the version naked on the command line */
	if (optind < argc && is_version_number(argv[optind]))
		version = strdup(argv[optind]);
	else {
		struct utsname buf;
		uname(&buf);
		version = strdup(buf.release);
	}

	/* Run old version if required. */
	if (old_module_version(version))
		exec_old_depmod(argv);

	/* Depmod -a by default if no names. */
	if (optind == argc)
		all = 1;

	dirname = malloc(strlen(basedir) + 1 + strlen(version) + 1
			 + sizeof("kernel"));
	sprintf(dirname, "%s/%s/kernel", basedir, version);

	/* If we're not stdout, open output files. */
	if (!depout) {
		/* Longest name to allocate */
		char depname[strlen(dirname) + 1
			    + sizeof("modules.generic_string")];

		sprintf(depname, "%s/%s/%s", basedir, version, "modules.dep");
		depout = fopen(depname, "w");
		if (!depout)
			fatal("Could not open %s for writing: %s\n",
			      depname, strerror(errno));

		sprintf(depname, "%s/%s/%s",
			basedir, version, "modules.pcimap");
		pciout = fopen(depname, "w");
		if (!pciout)
			fatal("Could not open %s for writing: %s\n",
			      depname, strerror(errno));

		sprintf(depname, "%s/%s/%s",
			basedir, version, "modules.usbmap");
		usbout = fopen(depname, "w");
		if (!usbout)
			fatal("Could not open %s for writing: %s\n",
			      depname, strerror(errno));
	}

	if (!all) {
		/* Do command line args. */
		for (opt = optind + 1; opt < argc; opt++)
			list = grab_module("", argv[opt], list);
	} else {
		list = grab_dir(dirname, list);
	}

	output_deps(list, skipchars, depout);
	output_pci_table(list, pciout);
	output_usb_table(list, usbout);

	free(dirname);
	free(version);
	
	return 0;
}
