static void *load_section64(int fd, unsigned long shdroff,
			    unsigned int num_secs,
			    unsigned int secnamesec,
			    const char *secname,
			    unsigned long *size)
{
	Elf64_Shdr sechdrs[num_secs];
	unsigned int i;
	char *secnames;

	/* Grab headers. */
	lseek(fd, shdroff, SEEK_SET);
	if (read(fd, sechdrs, sizeof(sechdrs)) != sizeof(sechdrs))
		return (void*)-1;

	/* Grab strings so we can tell who is who */
	secnames = malloc(sechdrs[secnamesec].sh_size);
	lseek(fd, sechdrs[secnamesec].sh_offset, SEEK_SET);
	if (read(fd, secnames, sechdrs[secnamesec].sh_size)
	    != sechdrs[secnamesec].sh_size) {
		free(secnames);
		return (void*)-1;
	}


	/* Find the section they want */
	for (i = 1; i < num_secs; i++) {
		if (strcmp(secnames+sechdrs[i].sh_name, secname) == 0) {
			void *buf;

			free(secnames);
			*size = sechdrs[i].sh_size;
			buf = malloc(*size);
			if (lseek(fd, sechdrs[i].sh_offset, SEEK_SET) == -1
			    || read(fd, buf, *size) != *size) {
				free(buf);
				return (void *)-1;
			}
			return buf;
		}
	}
	free(secnames);
	return NULL;
}

static void *map_exports64(int fd,
			   unsigned long shdroff,
			   unsigned int num_secs,
			   unsigned int secnamesec,
			   const char *name,
			   unsigned int *num_exports)
{
	struct kernel_symbol64 *syms;
	unsigned long size;

	syms = load_section64(fd, shdroff, num_secs, secnamesec, "__ksymtab",
			      &size);
	*num_exports = 0;
	if (syms == (void*)-1) {
		warn("Error finding exports for module %s\n", name);
		syms = NULL;
	} else if (syms)
		*num_exports = size / sizeof(struct kernel_symbol64);
	return syms;
}

static int export_name_cmp64(struct module *m, int index, const char *name)
{
	return strcmp(m->u.exports64[index].name, name);
}

static struct module *add_module64(int fd, struct module *new,
				   struct module *last,
				   struct alias **aliases)
{
	Elf64_Ehdr hdr;

	if (read(fd, &hdr, sizeof(hdr)) != sizeof(hdr)) {
		warn("Error reading module %s\n", new->name);
		free(new);
		close(fd);
		return last;
	}

	/* Map the section table */
	new->u.ptr = mops->map_exports(fd, hdr.e_shoff, hdr.e_shnum,
				       hdr.e_shstrndx, new->name,
				       &new->num_exports);
	/* Add in any aliases */
	*aliases = add_aliases(fd, hdr.e_shoff, hdr.e_shnum,
			       hdr.e_shstrndx, new, *aliases);

	close(fd);
	return new;
}

/* Analyse this module to see if it needs others. */
static int get_deps64(unsigned int order,
		      const char *dirname,
		      const char *modname,
		      struct module *modules,
		      int verbose)
{
	unsigned int i;
	unsigned long size;
	Elf64_Ehdr hdr;
	int fd;
	char *strings;
	Elf64_Sym *syms;
	int needed = 0;
	char modpath[strlen(dirname) + strlen(modname)
		    + sizeof(MODULE_EXTENSION)];

	sprintf(modpath, "%s%s%s", dirname, modname, MODULE_EXTENSION);

	fd = open(modpath, O_RDONLY);
	if (fd < 0)
		fatal("Can't open module %s: %s\n", modpath, strerror(errno));

	if (read(fd, &hdr, sizeof(hdr)) != sizeof(hdr))
		fatal("Error reading module %s\n", modpath);

	strings = mops->load_section(fd, hdr.e_shoff, hdr.e_shnum,
				     hdr.e_shstrndx, ".strtab", &size);
	syms = mops->load_section(fd, hdr.e_shoff, hdr.e_shnum,
				  hdr.e_shstrndx, ".symtab", &size);
	if (!strings || strings == (void *)-1
	    || !syms || syms == (void *)-1) {
		fatal("Could not load strings and symbol table from %s\n",
		      modpath);
	}

	/* Now establish which modules we need */
	for (i = 0; i < size / sizeof(syms[0]); i++) {
		if (syms[i].st_shndx == SHN_UNDEF) {
			/* Look for symbol */
			const char *name = strings + syms[i].st_name;

			if (strcmp(name, "") == 0)
				continue;

			/* Did this pull in a new module? */
			if (need_symbol(order, name, modules,
					verbose ? modpath : NULL))
				needed = 1;
		}
	}
	close(fd);
	return needed;
}

static struct mod_ops mod64_ops = {
	.load_section		=	load_section64,
	.map_exports		=	map_exports64,
	.export_name_cmp	=	export_name_cmp64,
	.add_module		=	add_module64,
	.get_deps		=	get_deps64,
};
