﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/greengrassv2/GreengrassV2Request.h>
#include <aws/greengrassv2/GreengrassV2_EXPORTS.h>
#include <aws/greengrassv2/model/RecipeOutputFormat.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace GreengrassV2 {
namespace Model {

/**
 */
class GetComponentRequest : public GreengrassV2Request {
 public:
  AWS_GREENGRASSV2_API GetComponentRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetComponent"; }

  AWS_GREENGRASSV2_API Aws::String SerializePayload() const override;

  AWS_GREENGRASSV2_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The format of the recipe.</p>
   */
  inline RecipeOutputFormat GetRecipeOutputFormat() const { return m_recipeOutputFormat; }
  inline bool RecipeOutputFormatHasBeenSet() const { return m_recipeOutputFormatHasBeenSet; }
  inline void SetRecipeOutputFormat(RecipeOutputFormat value) {
    m_recipeOutputFormatHasBeenSet = true;
    m_recipeOutputFormat = value;
  }
  inline GetComponentRequest& WithRecipeOutputFormat(RecipeOutputFormat value) {
    SetRecipeOutputFormat(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>
   * of the component version.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  GetComponentRequest& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}
 private:
  RecipeOutputFormat m_recipeOutputFormat{RecipeOutputFormat::NOT_SET};

  Aws::String m_arn;
  bool m_recipeOutputFormatHasBeenSet = false;
  bool m_arnHasBeenSet = false;
};

}  // namespace Model
}  // namespace GreengrassV2
}  // namespace Aws
