/**
 * Copyright © 2009 Red Hat, Inc.
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice (including the next
 *  paragraph) shall be included in all copies or substantial portions of the
 *  Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 *  DEALINGS IN THE SOFTWARE.
 */

/* Test relies on assert() */
#undef NDEBUG

#include <dix-config.h>

/*
 * Protocol testing for XIQueryPointer request.
 */
#include <stdint.h>
#include <X11/X.h>
#include <X11/Xproto.h>
#include <X11/extensions/XI2proto.h>

#include "Xi/handlers.h"

#include "inputstr.h"
#include "windowstr.h"
#include "scrnintstr.h"
#include "exevents.h"
#include "exglobals.h"

#include "protocol-common.h"

DECLARE_WRAP_FUNCTION(WriteToClient, void, ClientPtr client, int len, void *data);

extern ClientRec client_window;
static ClientRec client_request;
static void reply_XIQueryPointer_data(ClientPtr client, int len, void *data);

static struct {
    DeviceIntPtr dev;
    WindowPtr win;
} test_data;

static void
reply_XIQueryPointer(ClientPtr client, int len, void *data)
{
    xXIQueryPointerReply *repptr = (xXIQueryPointerReply *) data;
    xXIQueryPointerReply reply = *repptr; /* copy so swapping doesn't touch the real reply */
    SpritePtr sprite;

    assert(len < 0xffff); /* suspicious size, swapping bug */

    if (!reply.repType)
        return;

    if (client->swapped) {
        swapl(&reply.length);
        swaps(&reply.sequenceNumber);
        swapl(&reply.root);
        swapl(&reply.child);
        swapl(&reply.root_x);
        swapl(&reply.root_y);
        swapl(&reply.win_x);
        swapl(&reply.win_y);
        swaps(&reply.buttons_len);
    }

    reply_check_defaults(&reply, len, XIQueryPointer);

    assert(reply.root == root.drawable.id);
    assert(reply.same_screen == xTrue);

    sprite = test_data.dev->spriteInfo->sprite;
    assert((reply.root_x >> 16) == sprite->hot.x);
    assert((reply.root_y >> 16) == sprite->hot.y);

    if (test_data.win == &root) {
        assert(reply.root_x == reply.win_x);
        assert(reply.root_y == reply.win_y);
        assert(reply.child == window.drawable.id);
    }
    else {
        int x, y;

        x = sprite->hot.x - window.drawable.x;
        y = sprite->hot.y - window.drawable.y;

        assert((reply.win_x >> 16) == x);
        assert((reply.win_y >> 16) == y);
        assert(reply.child == None);
    }

    assert(reply.same_screen == xTrue);

    wrapped_WriteToClient = reply_XIQueryPointer_data;
}

static void
reply_XIQueryPointer_data(ClientPtr client, int len, void *data)
{
    wrapped_WriteToClient = reply_XIQueryPointer;

    assert(len < 0xffff); /* suspicious size, swapping bug */
}

static void
request_XIQueryPointer(ClientPtr client, xXIQueryPointerReq * req, int error)
{
    int rc;

    client_request.swapped = FALSE;
    rc = ProcXIQueryPointer(&client_request);
    assert(rc == error);

    if (rc == BadDevice)
        assert(client_request.errorValue == req->deviceid);

    client_request.swapped = TRUE;
    swaps(&req->deviceid);
    swapl(&req->win);
    swaps(&req->length);
    rc = ProcXIQueryPointer(&client_request);
    assert(rc == error);

    if (rc == BadDevice)
        assert(client_request.errorValue == req->deviceid);
}

static void
test_XIQueryPointer(void)
{
    int i;
    xXIQueryPointerReq request;

    init_simple();

    memset(&request, 0, sizeof(request));

    request_init(&request, XIQueryPointer);

    wrapped_WriteToClient = reply_XIQueryPointer;

    client_request = init_client(request.length, &request);

    request.deviceid = XIAllDevices;
    request_XIQueryPointer(&client_request, &request, BadDevice);

    request.deviceid = XIAllMasterDevices;
    request_XIQueryPointer(&client_request, &request, BadDevice);

    request.win = root.drawable.id;
    test_data.win = &root;

    test_data.dev = devices.vcp;
    request.deviceid = devices.vcp->id;
    request_XIQueryPointer(&client_request, &request, Success);
    request.deviceid = devices.vck->id;
    request_XIQueryPointer(&client_request, &request, BadDevice);
    request.deviceid = devices.mouse->id;
    request_XIQueryPointer(&client_request, &request, BadDevice);
    request.deviceid = devices.kbd->id;
    request_XIQueryPointer(&client_request, &request, BadDevice);

    test_data.dev = devices.mouse;
    devices.mouse->master = NULL;       /* Float, kind-of */
    request.deviceid = devices.mouse->id;
    request_XIQueryPointer(&client_request, &request, Success);

    for (i = devices.kbd->id + 1; i <= 0xFFFF; i++) {
        request.deviceid = i;
        request_XIQueryPointer(&client_request, &request, BadDevice);
    }

    request.win = window.drawable.id;

    test_data.dev = devices.vcp;
    test_data.win = &window;
    request.deviceid = devices.vcp->id;
    request_XIQueryPointer(&client_request, &request, Success);

    test_data.dev = devices.mouse;
    request.deviceid = devices.mouse->id;
    request_XIQueryPointer(&client_request, &request, Success);

    /* test REQUEST_SIZE_MATCH */
    client_request.req_len -= 4;
    request_XIQueryPointer(&client_request, &request, BadLength);
}

const testfunc_t*
protocol_xiquerypointer_test(void)
{
    static const testfunc_t testfuncs[] = {
        test_XIQueryPointer,
        NULL,
    };
    return testfuncs;
}
