// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import { needs, readOnlyProperty, cloneMaterial, } from '@aws-crypto/material-management';
import { Maximum } from '@aws-crypto/serialize';
export function decorateProperties(obj, input) {
    const { cache, backingMaterialsManager, maxAge, maxBytesEncrypted, maxMessagesEncrypted, partition, } = input;
    /* Precondition: A caching material manager needs a cache. */
    needs(cache, 'You must provide a cache.');
    /* Precondition: A caching material manager needs a way to get material. */
    needs(backingMaterialsManager, 'You must provide a backing material source.');
    /* Precondition: You *can not* cache something forever. */
    needs(maxAge > 0, 'You must configure a maxAge');
    /* Precondition: maxBytesEncrypted must be inside bounds.  i.e. positive and not more than the maximum. */
    needs(!maxBytesEncrypted ||
        (maxBytesEncrypted > 0 &&
            Maximum.BYTES_PER_CACHED_KEY_LIMIT >= maxBytesEncrypted), 'maxBytesEncrypted is outside of bounds.');
    /* Precondition: maxMessagesEncrypted must be inside bounds.  i.e. positive and not more than the maximum. */
    needs(!maxMessagesEncrypted ||
        (maxMessagesEncrypted > 0 &&
            Maximum.MESSAGES_PER_CACHED_KEY_LIMIT >= maxMessagesEncrypted), 'maxMessagesEncrypted is outside of bounds.');
    /* Precondition: partition must be a string. */
    needs(partition && typeof partition === 'string', 'partition must be a string.');
    readOnlyProperty(obj, '_cache', cache);
    readOnlyProperty(obj, '_backingMaterialsManager', backingMaterialsManager);
    readOnlyProperty(obj, '_maxAge', maxAge);
    readOnlyProperty(obj, '_maxBytesEncrypted', maxBytesEncrypted || Maximum.BYTES_PER_CACHED_KEY_LIMIT);
    readOnlyProperty(obj, '_maxMessagesEncrypted', maxMessagesEncrypted || Maximum.MESSAGES_PER_CACHED_KEY_LIMIT);
    readOnlyProperty(obj, '_partition', partition);
}
export function getEncryptionMaterials({ buildEncryptionMaterialCacheKey, }) {
    return async function getEncryptionMaterials(request) {
        const { suite, encryptionContext, plaintextLength, commitmentPolicy } = request;
        /* Check for early return (Postcondition): If I can not cache the EncryptionMaterial, do not even look. */
        if ((suite && !suite.cacheSafe) ||
            typeof plaintextLength !== 'number' ||
            plaintextLength < 0) {
            const material = await this._backingMaterialsManager.getEncryptionMaterials(request);
            return material;
        }
        const cacheKey = await buildEncryptionMaterialCacheKey(this._partition, {
            suite,
            encryptionContext,
        });
        const entry = this._cache.getEncryptionMaterial(cacheKey, plaintextLength);
        /* Check for early return (Postcondition): If I have a valid EncryptionMaterial, return it. */
        if (entry && !this._cacheEntryHasExceededLimits(entry)) {
            return cloneResponse(entry.response);
        }
        else {
            this._cache.del(cacheKey);
        }
        const material = await this._backingMaterialsManager
            /* Strip any information about the plaintext from the backing request,
             * because the resulting response may be used to encrypt multiple plaintexts.
             */
            .getEncryptionMaterials({ suite, encryptionContext, commitmentPolicy });
        /* Check for early return (Postcondition): If I can not cache the EncryptionMaterial, just return it. */
        if (!material.suite.cacheSafe)
            return material;
        /* It is possible for an entry to exceed limits immediately.
         * The simplest case is to need to encrypt more than then maxBytesEncrypted.
         * In this case, I return the response to encrypt the data,
         * but do not put a know invalid item into the cache.
         */
        const testEntry = {
            response: material,
            now: Date.now(),
            messagesEncrypted: 1,
            bytesEncrypted: plaintextLength,
        };
        if (!this._cacheEntryHasExceededLimits(testEntry)) {
            this._cache.putEncryptionMaterial(cacheKey, material, plaintextLength, this._maxAge);
            return cloneResponse(material);
        }
        else {
            /* Postcondition: If the material has exceeded limits it MUST NOT be cloned.
             * If it is cloned, and the clone is returned,
             * then there exist a copy of the unencrypted data key.
             * It is true that this data would be caught by GC, it is better to just not rely on that.
             */
            return material;
        }
    };
}
export function decryptMaterials({ buildDecryptionMaterialCacheKey, }) {
    return async function decryptMaterials(request) {
        const { suite } = request;
        /* Check for early return (Postcondition): If I can not cache the DecryptionMaterial, do not even look. */
        if (!suite.cacheSafe) {
            const material = await this._backingMaterialsManager.decryptMaterials(request);
            return material;
        }
        const cacheKey = await buildDecryptionMaterialCacheKey(this._partition, request);
        const entry = this._cache.getDecryptionMaterial(cacheKey);
        /* Check for early return (Postcondition): If I have a valid DecryptionMaterial, return it. */
        if (entry && !this._cacheEntryHasExceededLimits(entry)) {
            return cloneResponse(entry.response);
        }
        else {
            this._cache.del(cacheKey);
        }
        const material = await this._backingMaterialsManager.decryptMaterials(request);
        this._cache.putDecryptionMaterial(cacheKey, material, this._maxAge);
        return cloneResponse(material);
    };
}
export function cacheEntryHasExceededLimits() {
    return function cacheEntryHasExceededLimits({ now, messagesEncrypted, bytesEncrypted }) {
        const age = Date.now() - now;
        return (age > this._maxAge ||
            messagesEncrypted > this._maxMessagesEncrypted ||
            bytesEncrypted > this._maxBytesEncrypted);
    };
}
/**
 * I need to clone the underlying material.
 * Because when the Encryption SDK is done with material, it will zero it out.
 * Plucking off the material and cloning just that and then returning the rest of the response
 * can just be handled in one place.
 * @param material EncryptionMaterial|DecryptionMaterial
 * @return EncryptionMaterial|DecryptionMaterial
 */
function cloneResponse(material) {
    return cloneMaterial(material);
}
//# sourceMappingURL=data:application/json;base64,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