#  Please refer to the COPYRIGHT file of the profiling package for details.
#  SPDX-License-Identifier: MIT
#
# profiling: Line by line profiling and code coverage for GAP
#
# Declarations
#

#! @Chapter Functionality provided by the &profiling; package

#! @Section Reading line-by-line profiles

#! @Arguments filename
#! @Description
#!   Read <A>filename</A>, a line-by-line profile which was previously generated
#!   by &GAP;, using the <Ref Func="ProfileLineByLine" BookName="ref"/>
#!   or <Ref Func="CoverageLineByLine" BookName="ref"/> functions from core &GAP;.
#!   A parsed profile can be transformed into a human-readable form using either
#!   <Ref Func="OutputAnnotatedCodeCoverageFiles"/> or
#!   <Ref Func="OutputFlameGraph"/>
DeclareGlobalFunction( "ReadLineByLineProfile" );

#! @Arguments filenames
#! @Description
#!   Read <A>filenames</A>, a list of line-by-line profiles which were previously
#!   generated by &GAP;, using the <Ref Func="ProfileLineByLine" BookName="ref"/>
#!   or <Ref Func="CoverageLineByLine" BookName="ref"/> functions from core &GAP;.
#!   The elements of <A>filenames</A> can be either filenames,
#!   or files previously parsed by <Ref Func="ReadLineByLineProfile"/>.
DeclareGlobalFunction( "MergeLineByLineProfiles" );



#! @Section Generating flame graphs
#!
#! A 'flame graph' is a method of visualising where time is spent by a program.


#! @Arguments profile [, filename] [, options]
#! @Description
#!   Generate an 'svg' file which represents a 'flame graph', a method of visualising
#!   where time is spent by a program.
#!   <P/>
#!   <A>profile</A> should be either a profile previously read by
#!   <Ref Func="ReadLineByLineProfile"/>,
#!   or a string giving the filename of a profile.
#!   <P/>
#!   The flame graph will be written to <A>filename</A> (or returned as a
#!   string if <A>filename</A> is not present).
#!   <P/>
#!   The final (optional) argument is a record of options. Currently, the allowed
#!   options are 'squash' (which is a boolean). If 'squash' is true then recursive
#!   functions calls will be squashed, so the graph will not show recursive functions
#!   calling themselves. The other allowed option is 'type', which can be "default"
#!   (a standard flamegraph) or "reverse" (reverse the graph, showing the leaf functions
#!   first)
DeclareGlobalFunction("OutputFlameGraph");

#! @Arguments profile [, filename]
#! @Description
#!   Generate the input required to draw a 'flame graph', a method of visualising
#!   where time is spent by a program. One program for drawing flame graphs using
#!   this output can be found at <URL>https://github.com/brendangregg/FlameGraph</URL>.
#!   <P/>
#!   <A>profile</A> should be either a profile previously read by
#!   <Ref Func="ReadLineByLineProfile"/>,
#!   or a string giving the filename of a profile.
#!   <P/>
#!   The flame graph input will be written to <A>filename</A> (or returned as a
#!   string if <A>filename</A> is not present).
#!   <P/>
DeclareGlobalFunction("OutputFlameGraphInput");


#! @Section Generating coverage reports

#! @Arguments coverage, [indir,] outdir [, options]
#! @Description
#!   Takes a previously generated profile and outputs HTML which shows the lines
#!   of code that were executed, and (if this was originally recorded) how long was spent
#!   executing these lines of code.
#!   <P/>
#!   <A>coverage</A> should be either a profile previously read by
#!   <Ref Func="ReadLineByLineProfile"/>,
#!   or a string giving the filename of a profile which will first
#!   be read with <F>ReadLineByLineProfile</F>.
#!   <P/>
#!   Files will be written to the directory <A>outdir</A>.
#!   <P/>
#!   The optional second argument gives a filter, only information about filenames
#!   starting with <A>indir</A> will be outputted.
#!   <P/>
#!   The final optional argument is a record of configuration options. The only
#!   currently allowed option is 'title', which will set the title of created pages.
DeclareGlobalFunction("OutputAnnotatedCodeCoverageFiles");

#! @Arguments coverage, outfile
#! @Description
#!   Takes a previously generated profile and outputs a json coverage file which is
#!   amongst other things accepted by <URL>codecov.io</URL>.
#!   <P/>
#!   <A>coverage</A> should be either a profile previously read by
#!   <Ref Func="ReadLineByLineProfile"/>,
#!   or a string giving the filename of a profile which will first
#!   be read with <F>ReadLineByLineProfile</F>.
#!   <P/>
#!   The output will be written to the file with name <A>outfile</A> (a string).
#!   <P/>
DeclareGlobalFunction("OutputJsonCoverage");


#! @Arguments coverage, outfile
#! @Description
#!   Takes a previously generated profile and outputs an lcov coverage file.
#!   <P/>
#!   <A>coverage</A> should be either a profile previously read by
#!   <Ref Func="ReadLineByLineProfile"/>, or the filename of a profile which will first
#!   be read with <F>ReadLineByLineProfile</F>.
#!   <P/>
#!   The output will be written to the file with name <A>outfile</A> (a string).
#!   <P/>
DeclareGlobalFunction("OutputLcovCoverage");

#! @Arguments coverage, outfile, pathtoremove[, opt]
#! @Description
#!   Takes a previously generated profile and outputs a json coverage file which is
#!   accepted by <URL>coveralls.io</URL>.
#!   <P/>
#!   <A>coverage</A> should be either a profile previously read by
#!   <Ref Func="ReadLineByLineProfile"/>,
#!   or a string giving the filename of a profile which will first
#!   be read with <F>ReadLineByLineProfile</F>.
#!   <A>pathtoremove</A> is the path to the tested repository;
#!   this path prefix will be removed from all filenames in <A>coverage</A>.
#!   Finally, <A>opt</A> is a record. Its key/value pairs are directly inserted into
#!   the produced JSON, in the form of a JSON dictionary. This can be used to
#!   set the <C>service_name</C>, <C>service_job_id</C>, and more. If this record
#!   is not given, we try to guess the correct values based on the environment
#!   (currently only supported for Travis and AppVeyor).
#!   <P/>
#!   The output will be written to the file with name <A>outfile</A> (a string).
#!   <P/>
DeclareGlobalFunction("OutputCoverallsJsonCoverage");

#! @Section Miscellaneous

#! @Arguments function, arguments
#! @Description
#!   Calls <A>function</A> with the list of arguments <A>arguments</A>,
#!   and opens a time profile of the resulting call in the default web
#!   browser.
#!   <P/>
DeclareGlobalFunction("LineByLineProfileFunction");

#! @Arguments file[, opts]
#! @Returns
#!   a string
#! @Description
#!   Tests the file with name <A>file</A> in another &GAP; session, and produces a
#!   code coverage report of lines that were executed in the process.  If
#!   <A>file</A> ends with <C>.tst</C> it will be called with <C>Test</C>;
#!   otherwise, it will be run directly.
#!
#!   The optional argument <A>opts</A> should be a record, and may contain any
#!   of the following components:
#!     * <C>outdir</C>: a string denoting the directory into which the HTML
#!       files of the report will be placed (a temporary directory by default);
#!     * <C>indir</C>: a string such that only file paths beginning with
#!       <C>indir</C> will be profiled (default <C>""</C>);
#!     * <C>showOutput</C>: a boolean denoting whether to print test output to
#!       the screen (default <K>true</K>);
#!     * <C>open</C>: a boolean denoting whether to open the report in a web
#!       browser on completion (default <K>false</K>).
#!
#!   This function returns the location of an HTML file containing the report.
DeclareGlobalFunction("ProfileFile");

#! @Arguments pkg_name[, opts]
#! @Returns
#!   a string
#! @Description
#!   If <A>pkg_name</A> is the name of an installed package, then this function
#!   runs that package's test suite and produces a report on the code coverage
#!   of files inside the package.  The string returned denotes the location of
#!   an HTML file containing the report.  The optional argument <A>opts</A>
#!   behaves the same as in <Ref Func = "ProfileFile"/>.
DeclareGlobalFunction("ProfilePackage");
